<?php

namespace App\Filament\Pages;

use App\Services\Settings\SettingsManager;
use Filament\Forms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Pages\Page;
use Filament\Notifications\Notification;
use Filament\Facades\Filament;

class Settings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-cog';
    protected static ?string $navigationGroup = 'System';
    protected static ?int $navigationSort = 100;
    protected static string $view = 'filament.pages.settings';

    public array $data = [];

    public function mount(): void
    {
        $this->fillFormData();
    }

    protected function fillFormData(): void
    {
        $settings = app(SettingsManager::class);
        $config = config('settings.groups');

        // Get settings directly from database to ensure proper format
        $dbSettings = \App\Models\Setting::whereNotNull('group')
            ->where('key', 'not like', '%.%') // Exclude old dotted format keys
            ->get()
            ->keyBy(function ($setting) {
                return "{$setting->group}_{$setting->key}";
            });

        // Convert to the format expected by the form
        $this->data = [];

        // Go through all configured fields to ensure defaults are set
        foreach ($config as $groupKey => $groupData) {
            $groupId = $groupData['id'] ?? $groupKey;
            foreach ($groupData['sections'] ?? [] as $section) {
                foreach ($section['fields'] ?? [] as $field) {
                    $flatKey = "{$groupId}_{$field['key']}";

                    if ($dbSettings->has($flatKey)) {
                        // Use database value
                        $setting = $dbSettings->get($flatKey);
                        $value = $settings->cast($setting->value);

                        // Handle repeater fields - they need arrays, not JSON strings
                        if ($field['type'] === 'repeater') {
                            if (is_string($value)) {
                                $decoded = json_decode($value, true);
                                $value = is_array($decoded) ? $decoded : [];
                            }
                        }

                        $this->data[$flatKey] = $value;
                    } else {
                        // Use default value from config
                        $defaultValue = $field['default'] ?? null;

                        if ($field['type'] === 'repeater' && $defaultValue) {
                            // For repeater fields, decode JSON default
                            if (is_string($defaultValue)) {
                                $decoded = json_decode($defaultValue, true);
                                $defaultValue = is_array($decoded) ? $decoded : [];
                            }
                        }

                        $this->data[$flatKey] = $defaultValue;
                    }
                }
            }
        }

        $this->form->fill($this->data);
    }



    protected function getFormSchema(): array
    {
        $config = config('settings.groups');
        $tabs = [];

        foreach ($config as $groupKey => $group) {
            $groupId = $group['id'] ?? $groupKey;
            $sections = [];

            foreach ($group['sections'] ?? [] as $sectionKey => $section) {
                $fields = [];
                foreach ($section['fields'] ?? [] as $field) {
                    $flatKey = "{$groupId}_{$field['key']}";
                    $component = $this->buildFormComponent($field, $flatKey);
                    if ($component) {
                        $fields[] = $component;
                    }
                }

                if (!empty($fields)) {
                    $sections[] = Forms\Components\Section::make($section['label'])
                        ->description($section['description'] ?? null)
                        ->inlineLabel()
                        ->schema($fields);
                }
            }

            if (!empty($sections)) {
                $tabs[] = Forms\Components\Tabs\Tab::make($group['label'])
                    ->icon($group['icon'] ?? null)
                    ->schema($sections);
            }
        }

        return [
            Forms\Components\Tabs::make('Settings')
                ->tabs($tabs)
                ->columnSpan('full'),
        ];
    }

    protected function buildFormComponent(array $field, string $flatKey)
    {
        $component = match ($field['type']) {
            'text' => Forms\Components\TextInput::make($flatKey)
                ->label($field['label'])
                ->placeholder($field['placeholder'] ?? null),

            'number' => Forms\Components\TextInput::make($flatKey)
                ->label($field['label'])
                ->numeric()
                ->placeholder($field['placeholder'] ?? null),

            'textarea' => Forms\Components\Textarea::make($flatKey)
                ->label($field['label'])
                ->placeholder($field['placeholder'] ?? null)
                ->rows(3),

            'select' => $this->buildSelectComponent($field, $flatKey),

            'checkbox' => Forms\Components\CheckboxList::make($flatKey)
                ->label($field['label'])
                ->options($this->getFieldOptions($field)),

            'radio' => $this->buildRadioComponent($field, $flatKey),

            'toggle' => Forms\Components\Toggle::make($flatKey)
                ->label($field['label']),

            'file' => $this->buildFileUploadComponent($field, $flatKey),

            'password' => Forms\Components\TextInput::make($flatKey)
                ->label($field['label'])
                ->password()
                ->revealable(),

            'repeater' => $this->buildRepeaterComponent($field, $flatKey),

            default => null,
        };

        if ($component) {
            if (isset($field['rules'])) {
                $rules = is_string($field['rules']) ? explode('|', $field['rules']) : $field['rules'];
                if (in_array('required', $rules)) {
                    $component->required();
                }
            }

            if (isset($field['help'])) {
                $component->helperText($field['help']);
            }
        }

        return $component;
    }

    protected function buildRepeaterComponent(array $field, string $flatKey)
    {
        $repeater = Forms\Components\Repeater::make($flatKey)
            ->label($field['label'])
            ->schema($this->buildRepeaterSchema($field['schema'] ?? []))
            ->defaultItems(0)
            ->collapsible($field['collapsible'] ?? true)
            ->cloneable($field['cloneable'] ?? false)
            ->reorderable($field['reorderable'] ?? true)
            ->addActionLabel($field['addActionLabel'] ?? 'Add Item');

        if (isset($field['minItems'])) {
            $repeater->minItems($field['minItems']);
        }

        if (isset($field['maxItems'])) {
            $repeater->maxItems($field['maxItems']);
        }

        return $repeater;
    }

    protected function buildRepeaterSchema(array $fields): array
    {
        $schema = [];

        foreach ($fields as $field) {
            $component = match ($field['type']) {
                'text' => Forms\Components\TextInput::make($field['key'])
                    ->label($field['label'])
                    ->placeholder($field['placeholder'] ?? null),

                'textarea' => Forms\Components\Textarea::make($field['key'])
                    ->label($field['label'])
                    ->placeholder($field['placeholder'] ?? null)
                    ->rows(2),

                'select' => Forms\Components\Select::make($field['key'])
                    ->label($field['label'])
                    ->options($field['options'] ?? []),

                'toggle' => Forms\Components\Toggle::make($field['key'])
                    ->label($field['label']),

                'number' => Forms\Components\TextInput::make($field['key'])
                    ->label($field['label'])
                    ->numeric()
                    ->placeholder($field['placeholder'] ?? null),

                default => Forms\Components\TextInput::make($field['key'])
                    ->label($field['label'])
                    ->placeholder($field['placeholder'] ?? null),
            };

            if ($component) {
                if (isset($field['rules'])) {
                    $rules = is_string($field['rules']) ? explode('|', $field['rules']) : $field['rules'];
                    if (in_array('required', $rules)) {
                        $component->required();
                    }
                    if (in_array('numeric', $rules)) {
                        $component->numeric();
                    }
                }

                if (isset($field['help'])) {
                    $component->helperText($field['help']);
                }

                $schema[] = $component;
            }
        }

        return $schema;
    }

    protected function buildSelectComponent(array $field, string $flatKey)
    {
        $select = Forms\Components\Select::make($flatKey)
            ->label($field['label'])
            ->searchable()
            ->preload();

        $options = $this->getFieldOptions($field);

        if ($options !== null) {
            $select->options($options);
        }

        if (isset($field['options_type']) && $field['options_type'] === 'model') {
            $select->searchable();
            $select->preload();
        }

        return $select;
    }

    protected function buildRadioComponent(array $field, string $flatKey)
    {
        $radio = Forms\Components\Radio::make($flatKey)
            ->label($field['label']);

        $options = $this->getFieldOptions($field);
        $descriptions = $this->getFieldDescriptions($field);

        if ($options !== null) {
            $radio->options($options);
        }

        // Set inline if specified
        if (isset($field['inline']) && $field['inline']) {
            $radio->inline();
        }

        // Add descriptions if available
        if ($field['option_descriptions'] ?? false && !empty($descriptions)) {
            $radio->descriptions($descriptions);
        }

        return $radio;
    }

    protected function getFieldOptions(array $field)
    {
        $optionsType = $field['options_type'] ?? 'array';

        return match ($optionsType) {
            'array' => $field['options'] ?? [],

            'callback' => $this->getCallbackOptions($field['options']),

            'enum' => $this->getEnumOptions($field['options']),

            'model' => $this->getModelOptions($field),

            default => [],
        };
    }

    protected function getFieldDescriptions(array $field)
    {
        $optionsType = $field['options_type'] ?? 'array';

        return match ($optionsType) {
            'array' => $field['descriptions'] ?? [],

            'enum' => $this->getEnumDescriptions($field['options']),

            'callback', 'model' => [],

            default => [],
        };
    }

    protected function getCallbackOptions(string $callbackName)
    {
        $callbacks = config('settings.callbacks', []);

        if (isset($callbacks[$callbackName]) && is_callable($callbacks[$callbackName])) {
            return $callbacks[$callbackName]();
        }

        return [];
    }

    protected function getEnumOptions(string $enumClass)
    {
        if (!class_exists($enumClass)) {
            return [];
        }

        if (!enum_exists($enumClass)) {
            return [];
        }

        $options = [];
        foreach ($enumClass::cases() as $case) {
            $label = method_exists($case, 'getLabel') ? $case->getLabel() : $case->name;
            $options[$case->value ?? $case->name] = $label;
        }

        return $options;
    }

    protected function getEnumDescriptions(string $enumClass)
    {
        if (!class_exists($enumClass)) {
            return [];
        }

        if (!enum_exists($enumClass)) {
            return [];
        }

        $descriptions = [];
        foreach ($enumClass::cases() as $case) {
            if (method_exists($case, 'getDescription')) {
                $descriptions[$case->value ?? $case->name] = $case->getDescription();
            }
        }

        return $descriptions;
    }

    protected function getModelOptions(array $field)
    {
        $modelClass = $field['options'] ?? null;

        if (!$modelClass || !class_exists($modelClass)) {
            return [];
        }

        $labelColumn = $field['options_label'] ?? 'name';
        $valueColumn = $field['options_value'] ?? 'id';

        return $modelClass::pluck($labelColumn, $valueColumn)->toArray();
    }

    protected function getFormStatePath(): ?string
    {
        return 'data';
    }

    public function save(): void
    {
        $data = $this->form->getState();
        $settings = app(SettingsManager::class);
        $config = config('settings.groups');

        // Convert flat keys back to group/key pairs and save
        foreach ($data as $flatKey => $value) {
            // Find the group and key from config
            $found = false;
            foreach ($config as $groupKey => $group) {
                $groupId = $group['id'] ?? $groupKey;
                foreach ($group['sections'] ?? [] as $section) {
                    foreach ($section['fields'] ?? [] as $field) {
                        $expectedFlatKey = "{$groupId}_{$field['key']}";
                        if ($flatKey === $expectedFlatKey) {
                            $settings->set($groupId, $field['key'], is_array($value) ? json_encode($value) : $value, [
                                'type'  => $field['type'], // Use the actual field type from config
                                'created_by' => Filament::auth()->user()?->id,
                            ]);
                            $found = true;
                            break 3;
                        }
                    }
                }
            }

            if (!$found) {
                // Fallback for any fields not found in config
                $parts = explode('_', $flatKey, 2);
                if (count($parts) === 2) {
                    [$group, $key] = $parts;
                    $settings->set($group, $key, is_array($value) ? json_encode($value) : $value, [
                        'type'  => $this->inferType($value),
                        'created_by' => Filament::auth()->user()?->id,
                    ]);
                }
            }
        }

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();

        // Refill the form with updated data
        $this->fillFormData();
    }

    protected function inferType($value): string
    {
        return match (true) {
            is_bool($value)  => 'boolean',
            is_array($value) => 'json',
            default          => 'text',
        };
    }

    protected function buildFileUploadComponent(array $field, string $flatKey)
    {
        $component = Forms\Components\FileUpload::make($flatKey)
            ->label($field['label'])
            ->directory('settings')
            ->visibility('public');

        // Extract max file size from validation rules
        $maxSize = $this->extractMaxSizeFromRules($field['rules'] ?? '');
        if ($maxSize) {
            $component->maxSize($maxSize);
        }

        // Check if this field has image_options configuration
        $imageOptions = $field['image_options'] ?? null;

        if ($imageOptions) {
            // Mark as image field
            $component->image();

            // Apply accepted file types
            if (isset($imageOptions['accepted_types'])) {
                $component->acceptedFileTypes($imageOptions['accepted_types']);
            }

            // Apply avatar mode (circular display)
            if ($imageOptions['avatar'] ?? false) {
                $component->avatar();
            }

            // Apply image editor
            if ($imageOptions['image_editor'] ?? false) {
                $component->imageEditor();

                // Apply crop aspect ratio
                if (isset($imageOptions['crop_aspect_ratio'])) {
                    $component->imageCropAspectRatio($imageOptions['crop_aspect_ratio']);
                }

                // Apply circle cropper
                if ($imageOptions['circle_cropper'] ?? false) {
                    $component->imageEditorAspectRatios(['1:1'])
                        ->circleCropper();
                }

                // Apply viewport size
                if (isset($imageOptions['viewport_width']) && isset($imageOptions['viewport_height'])) {
                    $component->imageEditorViewportWidth($imageOptions['viewport_width'])
                        ->imageEditorViewportHeight($imageOptions['viewport_height']);
                }

                // Apply image editor mode
                if (isset($imageOptions['editor_mode'])) {
                    $component->imageEditorMode($imageOptions['editor_mode']);
                }

                // Apply image editor empty fill color
                if (isset($imageOptions['editor_empty_fill_color'])) {
                    $component->imageEditorEmptyFillColor($imageOptions['editor_empty_fill_color']);
                }
            }

            // Apply image resize mode
            if (isset($imageOptions['resize_mode'])) {
                $component->imageResizeMode($imageOptions['resize_mode']);
            }

            // Apply resize target dimensions
            if (isset($imageOptions['resize_target_width'])) {
                $component->imageResizeTargetWidth($imageOptions['resize_target_width']);
            }

            if (isset($imageOptions['resize_target_height'])) {
                $component->imageResizeTargetHeight($imageOptions['resize_target_height']);
            }

            // Apply image resize upscale
            if (isset($imageOptions['resize_upscale'])) {
                $component->imageResizeUpscale($imageOptions['resize_upscale']);
            }

            // Apply preview options
            if (isset($imageOptions['previewable'])) {
                $component->previewable($imageOptions['previewable']);
            }

            // Apply download option
            if ($imageOptions['downloadable'] ?? false) {
                $component->downloadable();
            }

            // Apply open in new tab option
            if ($imageOptions['openable'] ?? false) {
                $component->openable();
            }

            // Apply deletable option
            if (isset($imageOptions['deletable'])) {
                $component->deletable($imageOptions['deletable']);
            }

            // Note: WebP optimization would be handled at the server/storage level
            // as Filament doesn't have a built-in optimize() method
        } else {
            // Default image handling for file fields without specific options
            $component->image();
        }

        return $component;
    }

    /**
     * Extract max file size from validation rules string
     */
    protected function extractMaxSizeFromRules(string $rules): ?int
    {
        if (preg_match('/max:(\d+)/', $rules, $matches)) {
            return (int) $matches[1];
        }

        return null;
    }
}
