<?php

namespace App\Filament\Resources\ApiClient;

use App\Filament\Resources\ApiClient\Pages;
use App\Models\ApiClient;
use App\Services\ApiClient\ApiClientService;
use App\Enums\ApiClientType;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\Section;
use Filament\Notifications\Notification;

class ApiClientResource extends Resource
{
    protected static ?string $model = ApiClient::class;

    protected static ?string $slug = 'api-clients';

    protected static ?string $navigationIcon = 'heroicon-o-key';

    protected static ?string $navigationGroup = 'System';

    protected static ?string $navigationLabel = 'API Clients';

    protected static ?int $navigationSort = 99;

    // Hide from default navigation since we're adding it manually to Settings
    protected static bool $shouldRegisterNavigation = false;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('API Client Information')
                    ->description('Configure API client details and authentication settings')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('Client Name')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., Main Company Website or iOS App v1')
                            ->helperText('A descriptive name for this API client')
                            ->columnSpanFull(),

                        Forms\Components\Select::make('type')
                            ->label('Client Type')
                            ->required()
                            ->options(ApiClientType::options())
                            ->default(ApiClientType::Website->value)
                            ->native(false)
                            ->helperText('Type of application consuming the API')
                            ->columnSpan(1),

                        Forms\Components\Toggle::make('is_active')
                            ->label('Active Status')
                            ->default(true)
                            ->helperText('Only active clients can generate new tokens')
                            ->columnSpan(1),
                    ])
                    ->columns(2),

                Section::make('Token Information')
                    ->description('Token details are shown below. Note: Full tokens are only visible once upon creation.')
                    ->schema([
                        Forms\Components\Placeholder::make('tokens_info')
                            ->label('Active Tokens')
                            ->content(function (?ApiClient $record): string {
                                if (!$record) {
                                    return 'No tokens yet. Create the client first.';
                                }

                                $tokenCount = $record->tokens()->count();
                                return $tokenCount > 0
                                    ? "{$tokenCount} active token(s)"
                                    : 'No tokens generated yet.';
                            }),

                        Forms\Components\Repeater::make('token_list')
                            ->label('Token Details')
                            ->schema([
                                Forms\Components\TextInput::make('name')
                                    ->label('Token Name')
                                    ->disabled()
                                    ->columnSpan(2),

                                Forms\Components\TagsInput::make('abilities')
                                    ->label('Abilities')
                                    ->disabled()
                                    ->columnSpan(2),

                                Forms\Components\TextInput::make('last_used_at')
                                    ->label('Last Used')
                                    ->disabled()
                                    ->columnSpan(1),
                            ])
                            ->default(function (?ApiClient $record): array {
                                if (!$record) {
                                    return [];
                                }

                                $service = app(ApiClientService::class);
                                $stats = $service->getTokenStatistics($record->id);

                                return collect($stats['tokens'])->map(function ($token) {
                                    return [
                                        'name' => $token['name'],
                                        'abilities' => $token['abilities'],
                                        'last_used_at' => $token['last_used_at'] ?? 'Never',
                                    ];
                                })->toArray();
                            })
                            ->columns(5)
                            ->addable(false)
                            ->deletable(false)
                            ->reorderable(false)
                            ->columnSpanFull()
                            ->visible(fn (?ApiClient $record) => $record && $record->tokens()->count() > 0),

                        Forms\Components\Placeholder::make('created_info')
                            ->label('Created By')
                            ->content(function (?ApiClient $record): string {
                                if (!$record || !$record->creator) {
                                    return 'N/A';
                                }

                                return $record->creator->name . ' on ' . $record->created_at->format('M d, Y H:i A');
                            })
                            ->hidden(fn (?ApiClient $record) => !$record),
                    ])
                    ->columns(2)
                    ->visible(fn (?ApiClient $record) => $record !== null),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Client Name')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),

                Tables\Columns\BadgeColumn::make('type')
                    ->label('Type')
                    ->formatStateUsing(fn (ApiClientType $state): string => $state->getLabel())
                    ->color(fn (ApiClientType $state): string => $state->getColor())
                    ->icon(fn (ApiClientType $state): string => $state->getIcon())
                    ->sortable(),

                Tables\Columns\IconColumn::make('is_active')
                    ->label('Status')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->sortable(),

                Tables\Columns\TextColumn::make('tokens_count')
                    ->label('Active Tokens')
                    ->getStateUsing(fn (ApiClient $record) => $record->tokens()->count())
                    ->badge()
                    ->color('primary'),

                Tables\Columns\TextColumn::make('creator.name')
                    ->label('Created By')
                    ->sortable()
                    ->toggleable()
                    ->default('N/A'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->dateTime('M d, Y H:i A')
                    ->sortable()
                    ->toggleable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('type')
                    ->options(ApiClientType::options()),

                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('Status')
                    ->placeholder('All clients')
                    ->trueLabel('Active only')
                    ->falseLabel('Inactive only'),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),

                    Tables\Actions\Action::make('toggle_status')
                        ->label(fn (ApiClient $record) => $record->is_active ? 'Deactivate' : 'Activate')
                        ->icon(fn (ApiClient $record) => $record->is_active ? 'heroicon-o-x-circle' : 'heroicon-o-check-circle')
                        ->color(fn (ApiClient $record) => $record->is_active ? 'danger' : 'success')
                        ->action(function (ApiClient $record): void {
                            $service = app(ApiClientService::class);

                            try {
                                if ($record->is_active) {
                                    $service->deactivate($record->id);
                                    $status = 'deactivated';
                                } else {
                                    $service->activate($record->id);
                                    $status = 'activated';
                                }

                                Notification::make()
                                    ->title('Status Updated')
                                    ->body("API client has been {$status}.")
                                    ->success()
                                    ->send();
                            } catch (\Exception $e) {
                                Notification::make()
                                    ->title('Error')
                                    ->body($e->getMessage())
                                    ->danger()
                                    ->send();
                            }
                        })
                        ->requiresConfirmation(),

                    Tables\Actions\DeleteAction::make()
                        ->before(function (ApiClient $record): void {
                            // Revoke all tokens before deletion
                            $service = app(ApiClientService::class);
                            $service->revokeAllTokens($record->id);
                        }),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->before(function ($records): void {
                            // Revoke all tokens for all selected records
                            $service = app(ApiClientService::class);
                            foreach ($records as $record) {
                                $service->revokeAllTokens($record->id);
                            }
                        }),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListApiClients::route('/'),
            'create' => Pages\CreateApiClient::route('/create'),
            'edit' => Pages\EditApiClient::route('/{record}/edit'),
            'view' => Pages\ViewApiClient::route('/{record}'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('is_active', true)->count();
    }

    public static function getNavigationBadgeColor(): ?string
    {
        return 'success';
    }
}
