<?php

namespace App\Filament\Resources\ApiClient\Pages;

use App\Filament\Resources\ApiClient\ApiClientResource;
use App\Services\ApiClient\ApiClientService;
use App\DTOs\ApiClient\CreateApiClientDTO;
use App\DTOs\ApiClient\GenerateTokenDTO;
use Filament\Resources\Pages\CreateRecord;
use Filament\Notifications\Notification;

class CreateApiClient extends CreateRecord
{
    protected static string $resource = ApiClientResource::class;

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // Add created_by field
        $data['created_by'] = auth()->id();
        return $data;
    }

    protected function handleRecordCreation(array $data): \Illuminate\Database\Eloquent\Model
    {
        $service = app(ApiClientService::class);

        try {
            $dto = CreateApiClientDTO::fromArray($data);
            $apiClient = $service->create($dto);

            // Automatically generate a default token
            $this->generateDefaultToken($apiClient->id);

            return $apiClient;
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error Creating API Client')
                ->body($e->getMessage())
                ->danger()
                ->send();

            throw $e;
        }
    }

    protected function generateDefaultToken(int $apiClientId): void
    {
        $service = app(ApiClientService::class);

        try {
            $tokenDTO = GenerateTokenDTO::fromArray([
                'token_name' => 'default-token',
                'abilities' => ['*'], // Full access by default
            ]);

            $newToken = $service->generateToken($apiClientId, $tokenDTO);

            // Store token in session to display on view page
            session()->flash('generated_token', $newToken->plainTextToken);
            session()->flash('generated_token_name', 'default-token');

        } catch (\Exception $e) {
            Notification::make()
                ->title('Warning: Token Generation Failed')
                ->body('API Client created but token generation failed: ' . $e->getMessage())
                ->warning()
                ->send();
        }
    }

    protected function getRedirectUrl(): string
    {
        // Redirect to view page to show the created client with token
        return $this->getResource()::getUrl('view', ['record' => $this->record]);
    }

    protected function getCreatedNotificationTitle(): ?string
    {
        return 'API Client created successfully';
    }
}
