<?php

namespace App\Filament\Resources\ApiClient\Pages;

use App\Filament\Resources\ApiClient\ApiClientResource;
use App\Services\ApiClient\ApiClientService;
use App\Enums\ApiClientType;
use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\RepeatableEntry;

class ViewApiClient extends ViewRecord
{
    protected static string $resource = ApiClientResource::class;

    protected static string $view = 'filament.resources.api-client.pages.view-api-client';

    public $showTokenAlert = false;
    public $generatedToken = null;
    public $generatedTokenName = null;

    public function mount(int | string $record): void
    {
        parent::mount($record);

        // Check if there's a token in session (from create page)
        if (session()->has('generated_token')) {
            $this->showTokenAlert = true;
            $this->generatedToken = session('generated_token');
            $this->generatedTokenName = session('generated_token_name');

            // Clear session
            session()->forget(['generated_token', 'generated_token_name']);
        }
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('generate_token')
                ->label('Generate Token')
                ->icon('heroicon-o-key')
                ->color('success')
                ->form([
                    \Filament\Forms\Components\TextInput::make('token_name')
                        ->label('Token Name')
                        ->required()
                        ->placeholder('e.g., production-api-key')
                        ->helperText('A descriptive name for this token'),

                    \Filament\Forms\Components\CheckboxList::make('abilities')
                        ->label('Token Abilities')
                        ->options([
                            '*' => 'Full Access (All Abilities)',
                            'leads:read' => 'Read Leads',
                            'leads:create' => 'Create Leads',
                            'leads:update' => 'Update Leads',
                            'leads:delete' => 'Delete Leads',
                            'forms:read' => 'Read Forms',
                            'forms:create' => 'Create Forms',
                            'forms:update' => 'Update Forms',
                            'forms:delete' => 'Delete Forms',
                        ])
                        ->columns(2)
                        ->default(['*'])
                        ->required()
                        ->helperText('Select the permissions this token should have'),
                ])
                ->action(function (array $data): void {
                    $service = app(\App\Services\ApiClient\ApiClientService::class);

                    try {
                        $tokenDTO = \App\DTOs\ApiClient\GenerateTokenDTO::fromArray($data);
                        $newToken = $service->generateToken($this->record->id, $tokenDTO);

                        // Set properties to show token alert
                        $this->showTokenAlert = true;
                        $this->generatedToken = $newToken->plainTextToken;
                        $this->generatedTokenName = $data['token_name'];

                        \Filament\Notifications\Notification::make()
                            ->title('Token Generated Successfully')
                            ->body('The token is displayed below. Please copy and store it securely.')
                            ->success()
                            ->send();

                    } catch (\Exception $e) {
                        \Filament\Notifications\Notification::make()
                            ->title('Error Generating Token')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                })
                ->disabled(fn () => !$this->record->is_active)
                ->modalHeading('Generate New API Token')
                ->modalDescription('Create a new API token with specific abilities for this client.')
                ->modalSubmitActionLabel('Generate Token')
                ->modalWidth('2xl'),

            Actions\Action::make('revoke_all_tokens')
                ->label('Revoke All Tokens')
                ->icon('heroicon-o-no-symbol')
                ->color('danger')
                ->action(function (): void {
                    $service = app(\App\Services\ApiClient\ApiClientService::class);

                    try {
                        $service->revokeAllTokens($this->record->id);

                        \Filament\Notifications\Notification::make()
                            ->title('All Tokens Revoked')
                            ->body('All API tokens for this client have been revoked.')
                            ->success()
                            ->send();

                    } catch (\Exception $e) {
                        \Filament\Notifications\Notification::make()
                            ->title('Error Revoking Tokens')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                })
                ->requiresConfirmation()
                ->modalHeading('Revoke All Tokens')
                ->modalDescription('Are you sure you want to revoke all API tokens for this client? This action cannot be undone.')
                ->modalSubmitActionLabel('Revoke All')
                ->visible(fn () => $this->record->tokens()->count() > 0),

            Actions\EditAction::make(),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Section::make('API Client Information')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextEntry::make('name')
                                    ->label('Client Name')
                                    ->size('lg')
                                    ->weight('bold')
                                    ->icon('heroicon-o-building-office-2'),

                                IconEntry::make('is_active')
                                    ->label('Status')
                                    ->boolean()
                                    ->trueIcon('heroicon-o-check-circle')
                                    ->falseIcon('heroicon-o-x-circle')
                                    ->trueColor('success')
                                    ->falseColor('danger'),
                            ]),

                        Grid::make(2)
                            ->schema([
                                TextEntry::make('type')
                                    ->label('Client Type')
                                    ->badge()
                                    ->color(fn (ApiClientType $state): string => $state->getColor())
                                    ->icon(fn (ApiClientType $state): string => $state->getIcon())
                                    ->formatStateUsing(fn (ApiClientType $state): string => $state->getLabel()),

                                TextEntry::make('created_at')
                                    ->label('Created At')
                                    ->dateTime('M d, Y H:i A')
                                    ->icon('heroicon-o-calendar'),
                            ]),

                        Grid::make(1)
                            ->schema([
                                TextEntry::make('creator.name')
                                    ->label('Created By')
                                    ->default('N/A')
                                    ->icon('heroicon-o-user'),
                            ]),
                    ])
                    ->columns(1),

                Section::make('API Configuration')
                    ->description('Use this endpoint URL and generated tokens to integrate with your applications')
                    ->schema([
                        TextEntry::make('api_endpoint')
                            ->label('API Base URL')
                            ->getStateUsing(fn () => url('/api/v1'))
                            ->copyable()
                            ->copyMessage('API endpoint URL copied!')
                            ->copyMessageDuration(1500)
                            ->icon('heroicon-o-link')
                            ->color('primary')
                            ->weight('medium')
                            ->columnSpanFull(),
                    ])
                    ->collapsible()
                    ->persistCollapsed()
                    ->columns(1),

                Section::make('Token Statistics')
                    ->description('Token details are shown below. Note: Full tokens are only visible once upon creation.')
                    ->schema([
                        Grid::make(1)
                            ->schema([
                                RepeatableEntry::make('tokens')
                                    ->label('Active Tokens')
                                    ->getStateUsing(function ($record) {
                                        $service = app(ApiClientService::class);
                                        $stats = $service->getTokenStatistics($record->id);
                                        return $stats['tokens'];
                                    })
                                    ->schema([
                                        TextEntry::make('name')
                                            ->label('Token Name')
                                            ->weight('bold')
                                            ->columnSpan(2),

                                        TextEntry::make('abilities')
                                            ->label('Abilities')
                                            ->badge()
                                            ->getStateUsing(fn ($record): array => is_array($record['abilities']) ? $record['abilities'] : ['*'])
                                            ->columnSpan(2),

                                        TextEntry::make('last_used_at')
                                            ->label('Last Used')
                                            ->formatStateUsing(fn ($state) => $state ? \Carbon\Carbon::parse($state)->format('M d, Y H:i A') : 'Never')
                                            ->placeholder('Never'),

                                        TextEntry::make('created_at')
                                            ->label('Created')
                                            ->dateTime('M d, Y H:i A'),
                                    ])
                                    ->columns(6)
                                    ->columnSpanFull()
                                    ->placeholder('No tokens generated yet'),
                            ]),
                    ])
                    ->collapsible(),
            ]);
    }
}
