<?php

namespace App\Filament\Resources\Blog;

use App\Filament\Resources\Blog\PostCategoryResource\Pages;
use App\Models\Blog\PostCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Textarea;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Section;
use Illuminate\Support\Str;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\ColorPicker;
use App\Filament\Traits\HasSeoSection;
use Filament\Tables\Columns\ColorColumn;
use Filament\Tables\Actions\Action;


class PostCategoryResource extends Resource
{
    use HasSeoSection;

    protected static ?string $model = PostCategory::class;


    protected static ?string $navigationGroup = 'Blog';
    protected static ?string $navigationLabel = 'Categories';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Category')
                    ->tabs([
                        Tabs\Tab::make('Category')
                            ->schema([
                                TextInput::make('name')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                        if ($context === 'create' && empty($get('slug'))) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (?string $state, Set $set) {
                                        if ($state) {
                                            $set('slug', Str::slug($state));
                                        }
                                    })
                                    ->dehydrated()
                                    ->unique(ignoreRecord: true),
                                Select::make('parent_id')
                                    ->label('Parent Category')
                                    ->relationship(
                                        'parent',
                                        'name',
                                        fn(Builder $query) => $query->where('is_active', true)
                                    )
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent category')
                                    ->nullable(),
                                ColorPicker::make('color')
                                    ->label('Category Color')
                                    ->nullable(),
                                RichEditor::make('description')
                                    ->columnSpanFull(),
                                Toggle::make('is_active')
                                    ->label('Visible')
                                    ->default(true),
                            ]),
                        Tabs\Tab::make('SEO')
                            ->schema([
                                static::getSeoSection(),
                            ]),
                    ])
                    ->columnSpanFull()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('parent.name')
                    ->label('Parent')
                    ->searchable()
                    ->sortable(false),
                TextColumn::make('slug')
                    ->searchable()
                    ->sortable(),
                ColorColumn::make('color')
                    ->label('Color'),
                TextColumn::make('posts_count')
                    ->label('Count')
                    ->sortable(),
                ToggleColumn::make('is_active')
                    ->label('Visible'),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('post_categories.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('post_post_category')
                                ->selectRaw('COUNT(*)')
                                ->whereColumn('post_post_category.post_category_id', 'post_categories.id');
                        },
                        'posts_count' // Name of the additional column for the count
                    );
                return $query;
            })
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\Filter::make('is_active')
                    ->label('Is Visible')
                    ->toggle()
                    ->query(fn(Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        ->url(fn(PostCategory $record): string => route('journal.search', ['category' => $record->slug]))
                        ->hidden(fn(PostCategory $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Tables\Actions\EditAction::make()
                        ->slideOver()
                        ->modalWidth('3xl')
                        ->hidden(fn(PostCategory $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->excludeAttributes(['posts_count'])
                        ->hidden(fn(PostCategory $record): bool => $record->trashed()),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(PostCategory $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),

                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPostCategories::route('/'),
        ];
    }
}
