<?php

namespace App\Filament\Resources\Blog;

use App\Enums\PostStatus;
use App\Filament\Imports\Blog\PostImporter;
use App\Filament\Resources\Blog\PostResource\Pages;
use App\Filament\Traits\HasSeoSection;
use App\Models\Blog\Post;
use App\Models\User\User;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\ImportAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use FilamentTiptapEditor\Enums\TiptapOutput;
use FilamentTiptapEditor\TiptapEditor;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use FilamentTiptapEditor\Enums\TiptapHeadingLevels;

class PostResource extends Resource
{
    use HasSeoSection;

    protected static ?string $model = Post::class;

    // protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'Blog';

    protected static ?int $navigationSort = 1;

    protected static ?string $recordTitleAttribute = 'post_title';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // Adjusting to a 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7]) // Change to 7 for 70%
                            ->schema([
                                Section::make('Post Details')
                                    ->schema([
                                        TextInput::make('post_title')
                                            ->label('Title')
                                            ->placeholder('Enter the post title')
                                            ->required()
                                            ->autofocus()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                                if ($context === 'create' && empty($get('slug'))) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('slug')
                                            ->label('Slug')
                                            ->placeholder('Enter the post slug')
                                            ->helperText('Will be automatically generated from title if left empty.')
                                            ->maxLength(255)
                                            ->required()
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (?string $state, Set $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            })
                                            ->dehydrated(),
                                        Textarea::make('post_excerpt')
                                            ->label('Excerpt')
                                            ->placeholder('Enter a brief excerpt of the post')
                                            ->helperText('A short summary of the post content. If left empty, the first few lines of the content will be used.')
                                            ->rows(3)
                                            ->maxLength(400)
                                            ->reactive()
                                            ->hint(function ($state, Textarea $component) {
                                                $length = strlen($state ?? '');

                                                return $length > 0 ? "({$length}/{$component->getMaxLength()})" : null;
                                            }),
                                        TiptapEditor::make('post_description')
                                            ->label('Content')
                                            ->placeholder('Enter the post content')
                                            ->profile('default')
                                            ->disk('media') // optional, defaults to config setting
                                            ->directory('blog') // optional, defaults to config setting
                                            ->maxSize(2024) // optional, defaults to config setting
                                            ->output(TiptapOutput::Json) // optional, change the format for saved data, default is html
                                            ->maxContentWidth('6xl')
                                            ->required()
                                            ->extraInputAttributes(['style' => 'max-height: 50rem; min-height: 20rem;']),
                                    ]),
                                // SEO Section
                                static::getSeoSection(),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3]) // Change to 3 for 30%
                            ->schema([
                                Section::make('Settings')
                                    ->schema([
                                        Select::make('post_status')
                                            ->label('Post Status')
                                            ->helperText('Set the current status of your post')
                                            ->options([
                                                PostStatus::Draft->value => PostStatus::Draft->name,
                                                PostStatus::Published->value => PostStatus::Published->name,
                                                PostStatus::Pending->value => PostStatus::Pending->name,
                                                PostStatus::Expired->value => PostStatus::Expired->name,
                                                PostStatus::Scheduled->value => PostStatus::Scheduled->name,
                                            ])
                                            ->required()
                                            ->live()
                                            ->default(PostStatus::Draft->value)
                                            ->afterStateUpdated(function ($state, Set $set) {
                                                if ($state === PostStatus::Scheduled->value) {
                                                    $set('published_at', now()->addHour());
                                                }
                                            }),
                                        DateTimePicker::make('published_at')
                                            ->label('Published Date')
                                            ->seconds(false)
                                            ->visible(fn(Get $get): bool => in_array($get('post_status'), [
                                                PostStatus::Scheduled->value,
                                            ]))
                                            ->required(fn(Get $get): bool => $get('post_status') === PostStatus::Scheduled->value)
                                            ->minDate(fn(Get $get): ?\DateTime => $get('post_status') === PostStatus::Scheduled->value ? now() : null)
                                            ->helperText(fn(Get $get): string => $get('post_status') === PostStatus::Scheduled->value
                                                ? 'Select when this post should be published'
                                                : 'The date when this post was published'),
                                        Select::make('created_by')
                                            ->options(User::all()->pluck('name', 'id'))
                                            ->label('Author')
                                            ->preload()
                                            ->searchable()
                                            ->required()
                                            ->default(auth()->guard('web')->user()?->id),
                                        SpatieMediaLibraryFileUpload::make('featured_image')
                                            ->collection('featured_image')
                                            ->imageEditor()
                                            ->hintIcon('heroicon-m-photo')
                                            ->conversion('thumb'),
                                        Toggle::make('is_featured')
                                            ->label('Set as Featured Post')
                                            ->helperText('Show this post in featured sections')
                                            ->inline(true)
                                            ->onColor('success'),
                                        Select::make('categories')
                                            ->multiple()
                                            ->relationship(
                                                name: 'Categories',
                                                titleAttribute: 'name',
                                                modifyQueryUsing: fn(Builder $query) => $query
                                                    ->select(['post_categories.id', 'post_categories.name'])
                                            )
                                            ->createOptionForm([
                                                TextInput::make('name')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->unique('post_categories', 'name'),
                                            ])
                                            ->createOptionAction(
                                                fn(FormAction $action) => $action
                                                    ->modalWidth('sm')
                                                    ->modalHeading('Create Category')
                                            )
                                            ->preload()
                                            ->searchable()
                                            ->optionsLimit(100)
                                            ->loadingMessage('Loading categories...')
                                            ->noSearchResultsMessage('No categories found.')
                                            ->placeholder('Select categories')
                                            ->helperText('Choose one or more categories'),
                                        Select::make('tags')
                                            ->multiple()
                                            ->relationship(
                                                name: 'Tags',
                                                titleAttribute: 'name',
                                                modifyQueryUsing: fn(Builder $query) => $query
                                                    ->select(['post_tags.id', 'post_tags.name'])
                                            )
                                            ->createOptionForm([
                                                TextInput::make('name')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->unique('post_tags', 'name'),
                                            ])
                                            ->createOptionAction(
                                                fn(FormAction $action) => $action
                                                    ->modalWidth('sm')
                                                    ->modalHeading('Create Tag')
                                            )
                                            ->preload()
                                            ->searchable()
                                            ->optionsLimit(100)
                                            ->loadingMessage('Loading tags...')
                                            ->noSearchResultsMessage('No tags found.')
                                            ->placeholder('Select tags')
                                            ->helperText('Choose one or more tags'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->recordUrl(fn(Post $record) => $record->trashed() ? null : Pages\EditPost::getUrl(['record' => $record]))
            ->headerActions([
                ImportAction::make()
                    ->importer(PostImporter::class)
                    ->icon('heroicon-o-arrow-up-tray')
                    ->label('Import from CSV')
                    ->color('success'),
            ])
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('post_title')
                    ->label('Title')
                    ->tooltip(fn(Post $record): string => $record->post_title)
                    ->wrap()
                    ->searchable()
                    ->formatStateUsing(fn(Post $record): string => Str::limit($record->post_title, 50)),
                TextColumn::make('author.name')
                    ->label('Author')
                    ->searchable(),
                TextColumn::make('categories.name')
                    ->wrap(),
                IconColumn::make('is_featured')
                    ->label('Featured'),
                TextColumn::make('post_status')
                    ->badge()
                    ->label('Status'),
                TextColumn::make('created_at')
                    ->label('Date')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->getStateUsing(function (?Post $record): ?string {
                        if (! $record) {
                            return null;
                        }

                        return $record->post_status->value === PostStatus::Published->value
                            ? $record->published_at
                            : $record->created_at;
                    }),
            ])
            ->filters([
                // TrashedFilter::make(),
                SelectFilter::make('created_by')
                    ->label('Author')
                    ->relationship('author', 'name')
                    ->searchable()
                    ->preload()
                    ->multiple(),
                SelectFilter::make('categories')
                    ->relationship(
                        name: 'Categories',
                        titleAttribute: 'name',
                        modifyQueryUsing: fn(Builder $query) => $query
                            ->select(['post_categories.id', 'post_categories.name'])
                    )
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                TernaryFilter::make('is_featured')
                    ->label('Featured Posts'),
                SelectFilter::make('created_at')
                    ->label('Date')
                    ->options(function () {
                        $dates = Post::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                            ->distinct()
                            ->pluck('month')
                            ->map(function ($date) {
                                return [
                                    $date => date('F Y', strtotime($date . '-01')),
                                ];
                            })
                            ->collapse()
                            ->toArray();

                        return $dates;
                    })
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['value'],
                            fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                        );
                    }),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->filtersTriggerAction(
                fn(Action $action) => $action
                    ->button()
                    ->label('Filters'),
            )
            ->actions([
                ActionGroup::make([
                    Action::make('preview')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->label(
                            fn(Post $record): string => $record->post_status === PostStatus::Published
                                ? 'View'
                                : 'Preview'
                        )
                        ->url(fn(Post $record): string => route('journal.show', [
                            'slug' => $record->slug,
                            'preview' => $record->post_status !== PostStatus::Published ? true : null,
                        ]), true)
                        ->hidden(
                            fn(Post $record): bool => ! in_array($record->post_status->value, [
                                PostStatus::Published->value,
                                PostStatus::Scheduled->value,
                                PostStatus::Pending->value,
                                PostStatus::Draft->value,
                            ])
                        ),
                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Post $record): bool => $record->post_status->value === PostStatus::Pending->value
                        )
                        ->action(function (Post $record): void {
                            $record->post_status = PostStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Post')
                        ->modalDescription('Are you sure you want to approve this post? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve post')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post Approved')
                                ->body('Post has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(Post $record): bool => in_array($record->post_status->value, [PostStatus::Draft->value, PostStatus::Expired->value])
                        )
                        ->action(function (Post $record): void {
                            $record->post_status = PostStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Post')
                        ->modalDescription('Are you sure you want to publish this post? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish Post')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post Published')
                                ->body('Post has been published successfully')
                        ),

                    EditAction::make()
                        ->hidden(fn(Post $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->hidden(fn(Post $record): bool => $record->trashed())
                        ->beforeReplicaSaved(function (Post $replica): void {
                            $replica->post_status = PostStatus::Draft->value;
                            $replica->is_featured = false;
                        })
                        ->after(function (Post $replica, Post $record): void {
                            $replica->categories()->sync($record->categories->pluck('id'));
                            $replica->tags()->sync($record->tags->pluck('id'));
                        }),
                    DeleteAction::make()
                        ->hidden(fn(Post $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPosts::route('/'),
            'create' => Pages\CreatePost::route('/create'),
            'edit' => Pages\EditPost::route('/{record}/edit'),
        ];
    }

    public static function getGlobalSearchResultDetails(mixed $record): array
    {
        return [
            'Author' => $record->author->name,
            'Category' => $record->categories->pluck('name')->implode(', '),
        ];
    }

    public static function getNavigationIcon(): ?string
    {
        return 'heroicon-o-document-text';
    }

    public static function getImporters(): array
    {
        return [
            PostImporter::class,
        ];
    }
}
