<?php

namespace App\Filament\Resources\Blog\PostResource\Pages;

use App\Filament\Resources\Blog\PostResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use Filament\Notifications\Notification;
use App\Enums\PostStatus;
use Filament\Support\Facades\FilamentView;
use Illuminate\View\View;
use Filament\Actions\Action;
use Filament\Support\Exceptions\Halt;

class CreatePost extends CreateRecord
{
    protected static string $resource = PostResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'Save',
                    PostStatus::Scheduled->value => 'Schedule',
                    default => 'Publish',
                })
                ->icon(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'heroicon-o-document-text',
                    PostStatus::Scheduled->value => 'heroicon-o-clock',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'primary',
                    PostStatus::Scheduled->value => 'warning',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();

                    if ($data['post_status'] === PostStatus::Draft->value || $data['post_status'] === PostStatus::Published->value) {
                        $this->data['post_status'] = PostStatus::Published->value;
                        $this->data['published_at'] = now();
                    } elseif ($data['post_status'] === PostStatus::Scheduled->value && empty($data['published_at'])) {
                        $this->data['published_at'] = now()->addHour();
                    }

                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['post_status'] = PostStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn() => PostResource::getUrl()),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn() => match ($this->data['post_status'] ?? null) {
                    PostStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    if ($this->data['post_status'] === PostStatus::Draft->value || $this->data['post_status'] === PostStatus::Published->value) {
                        $this->data['post_status'] = PostStatus::Published->value;
                        $this->data['published_at'] = now();
                    }

                    $this->create();
                })
                ->keyBindings(['mod+s']),
            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['post_status'] = PostStatus::Draft->value;
                    $this->create();
                }),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // If the post status is set to Published, set the published_at timestamp to now
        if ($data['post_status'] === PostStatus::Published->value) {
            $data['published_at'] = now();
        }
        // If the post is scheduled but no publish date was set, default to 1 hour from now
        elseif ($data['post_status'] === PostStatus::Scheduled->value && empty($data['published_at'])) {
            $data['published_at'] = now()->addHour();
        }

        return $data;
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }

    protected function getCreatedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['post_status']) {
                PostStatus::Draft->value => 'Draft Saved',
                PostStatus::Pending->value => 'Post Saved',
                PostStatus::Scheduled->value => 'Post Scheduled',
                default => 'Post Published',
            })
            ->body(match ($this->data['post_status']) {
                PostStatus::Draft->value => 'Your post has been saved as draft.',
                PostStatus::Pending->value => 'Your post has been saved as pending.',
                PostStatus::Scheduled->value => 'Your post has been scheduled for publication.',
                default => 'Your post has been published successfully.',
            });
    }
}
