<?php

namespace App\Filament\Resources\Blog\PostResource\Pages;

use App\Enums\PostStatus;
use App\Filament\Resources\Blog\PostResource;
use App\Models\Blog\Post;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use Illuminate\View\View;

class EditPost extends EditRecord
{
    protected static string $resource = PostResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['post_status']) {
                PostStatus::Draft->value => 'Draft Updated',
                PostStatus::Pending->value => 'Post Updated',
                PostStatus::Scheduled->value => 'Schedule Updated',
                default => 'Post Updated',
            })
            ->body(match ($this->data['post_status']) {
                PostStatus::Draft->value => 'Your post has been updated and saved as draft.',
                PostStatus::Pending->value => 'Your post has been updated and saved as pending.',
                PostStatus::Scheduled->value => 'Your post schedule has been updated.',
                default => 'Your post has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),

            Actions\Action::make('preview')
                ->label('Preview')
                ->icon('heroicon-o-eye')
                ->color('success')
                ->url(fn() => route('journal.show', [
                    'slug' => $this->record->slug,
                    'preview' => $this->record->post_status !== PostStatus::Published ? true : null,
                ]))
                ->openUrlInNewTab(),

            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of actions
                    Actions\Action::make('history')
                        ->label('View History')
                        ->icon('heroicon-o-clock')
                        ->color('info')
                        ->action(fn(Post $record) => $record)
                        ->modalContent(view('filament.resources.includes.history', ['record' => $this->record]))
                        ->modalWidth('3xl')
                        ->modalHeading(fn(Post $record) => 'History for ' . $record->post_title)
                        ->modalSubmitAction(false),

                    Actions\DeleteAction::make()
                        ->label('Move to Trash')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->hidden(fn(): bool => $this->record->trashed())
                        ->modalHeading('Delete Post')
                        ->modalDescription('Are you sure you want to delete this post? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, delete it')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post deleted')
                                ->body('The post has been moved to trash.')
                        ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn() => PostResource::getUrl()),
            ])
                ->tooltip('More Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('info')
                ->button()
                ->extraAttributes([
                    'class' => 'more-actions-btn',
                ]),
        ];
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['post_status' => PostStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['post_status' => PostStatus::Draft]);
    }

    protected function afterSave(): void
    {
        if ($this->record->post_status !== PostStatus::Trashed && $this->record->trashed()) {
            $this->record->restore();
        }
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),

            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
