<?php

namespace App\Filament\Resources\Blog\PostResource\Pages;

use App\Filament\Resources\Blog\PostResource;
use App\Enums\PostStatus;
use App\Models\Blog\Post;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Filament\Resources\Components\Tab;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Cache;
use Filament\Support\Facades\FilamentView;
use Illuminate\View\View;

class ListPosts extends ListRecords
{
    protected static string $resource = PostResource::class;

    protected static string $view = 'vendor.filament.components.table.table-body';

    protected function getHeaderActions(): array
    {
        return [
            Actions\CreateAction::make()
                ->label('Add New')
                ->color('primary')
                ->icon('heroicon-o-plus'),
        ];
    }

    public function getTabs(): array
    {
        // Get cached counts
        $cachedCounts = Cache::remember('post-status-counts', 300, function () {
            $counts = Post::query()
                ->select('post_status')
                ->selectRaw('COUNT(*) as count')
                ->groupBy('post_status')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->post_status->value => $item->count];
                })
                ->toArray();

            $trashedCount = Post::onlyTrashed()->count();
            $allCount = array_sum($counts) + $trashedCount;

            return [
                'counts' => $counts,
                'trashedCount' => $trashedCount,
                'allCount' => $allCount,
            ];
        });

        $tabs = [];

        // Always show All tab
        $tabs['all'] = Tab::make('All')
            ->icon('heroicon-m-squares-2x2')
            ->badge($cachedCounts['allCount'])
            ->modifyQueryUsing(fn (Builder $query) => $query);

        // Only show Published tab if there are published posts
        if (($cachedCounts['counts'][PostStatus::Published->value] ?? 0) > 0) {
            $tabs['published'] = Tab::make('Published')
                ->icon('heroicon-m-check-circle')
                ->badge($cachedCounts['counts'][PostStatus::Published->value])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('post_status', PostStatus::Published));
        }

        // Only show Scheduled tab if there are scheduled posts
        if (($cachedCounts['counts'][PostStatus::Scheduled->value] ?? 0) > 0) {
            $tabs['scheduled'] = Tab::make('Scheduled')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][PostStatus::Scheduled->value])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('post_status', PostStatus::Scheduled));
        }

        // Only show Pending tab if there are pending posts
        if (($cachedCounts['counts'][PostStatus::Pending->value] ?? 0) > 0) {
            $tabs['pending'] = Tab::make('Pending Review')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][PostStatus::Pending->value])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('post_status', PostStatus::Pending));
        }

        // Only show Draft tab if there are draft posts
        if (($cachedCounts['counts'][PostStatus::Draft->value] ?? 0) > 0) {
            $tabs['draft'] = Tab::make('Draft')
                ->icon('heroicon-m-document')
                ->badge($cachedCounts['counts'][PostStatus::Draft->value])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('post_status', PostStatus::Draft));
        }

        if (($cachedCounts['counts'][PostStatus::Expired->value] ?? 0) > 0) {
            $tabs['expired'] = Tab::make('Expired')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][PostStatus::Expired->value])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('post_status', PostStatus::Expired));
        }

        // Only show Trash tab if there are trashed posts
        if ($cachedCounts['trashedCount'] > 0) {
            $tabs['trash'] = Tab::make('Trash')
                ->icon('heroicon-m-trash')
                ->badge($cachedCounts['trashedCount'])
                ->modifyQueryUsing(fn (Builder $query) => $query->withTrashed()->whereNotNull('deleted_at'));
        }

        return $tabs;
    }

    public function getDefaultActiveTab(): string | int | null
    {
        return 'all';
    }

    protected function getTableRecordsPerPageSelectOptions(): array
    {
        return [10, 25, 50];
    }

    protected function getTableQuery(): Builder
    {
        return parent::getTableQuery()
            ->with(['media', 'author', 'categories'])
            ->latest();
    }

    // protected function getHeaderActionsPosition(): string 
    // {
    //     return 'content-start';
    // }
}
