<?php

namespace App\Filament\Resources\Blog\PostTagResource\Pages;

use App\Filament\Resources\Blog\PostTagResource;
use App\Models\Blog\PostTag;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PostTags extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = PostTagResource::class;

    protected static string $view = 'filament.resources.blog.post-tag-resource.pages.post-tags';

    public ?array $data = [];

    public ?PostTag $postTag = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (! $this->postTag && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                RichEditor::make('description'),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(PostTag::query())
            ->columns([
                TextColumn::make('name')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('posts_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        ->url(fn(PostTag $record): string => route('journal.tag', ['slug' => $record->slug]))
                        ->hidden(fn(PostTag $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(PostTag $record): bool => $record->trashed())
                        ->action(function (PostTag $record): void {
                            $this->postTag = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['posts_count'])
                        ->hidden(fn(PostTag $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(PostTag $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),

                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('post_tags.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('post_post_tag')
                                ->selectRaw('COUNT(*)')
                                ->whereColumn('post_post_tag.post_tag_id', 'post_tags.id');
                        },
                        'posts_count'
                    );

                return $query;
            });
    }

    public function save(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $data = $this->form->getState();

            if ($this->postTag) {
                $this->postTag->update($data);
                $message = 'Tag updated successfully';
            } else {
                PostTag::create($data);
                $message = 'Tag created successfully';
            }

            DB::commit();

            $this->postTag = null;
            $this->data = [];
            $this->form->fill();

            Notification::make()
                ->success()
                ->title($message)
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->danger()
                ->title('Error saving tag')
                ->body('Please try again.')
                ->send();
        }
    }

    public function cancelEdit(): void
    {
        $this->postTag = null;
        $this->data = [];
        $this->form->fill();
    }
}
