<?php

namespace App\Filament\Resources\Classified;

use App\Filament\Resources\Classified\ClassifiedCategoryResource\Pages;
use App\Models\Classified\ClassifiedCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Textarea;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Str;

class ClassifiedCategoryResource extends Resource
{
    protected static ?string $model = ClassifiedCategory::class;

    protected static ?string $navigationGroup = 'Classified';
    protected static ?string $navigationLabel = 'Categories';
    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Category')
                    ->tabs([
                        Tabs\Tab::make('Category')
                            ->schema([
                                TextInput::make('classified_category_name')
                                    ->label('Category Name')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                        if ($context === 'create' && empty($get('slug'))) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (?string $state, Set $set) {
                                        if ($state) {
                                            $set('slug', Str::slug($state));
                                        }
                                    })
                                    ->dehydrated()
                                    ->unique(ignoreRecord: true),
                                Select::make('parent_id')
                                    ->label('Parent Category')
                                    ->relationship(
                                        'parent',
                                        'classified_category_name',
                                        fn (Builder $query) => $query->where('is_active', true)
                                    )
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent category')
                                    ->nullable(),
                                RichEditor::make('classified_category_description')
                                    ->label('Description')
                                    ->disableToolbarButtons([
                                        'codeBlock',
                                    ])
                                    ->columnSpanFull(),
                                Toggle::make('is_active')
                                    ->label('Visible')
                                    ->default(true),
                            ]),
                        Tabs\Tab::make('SEO')
                            ->schema([
                                Section::make()
                                    ->relationship('seo')
                                    ->schema([
                                        TextInput::make('title')
                                            ->label('Meta Title')
                                            ->maxLength(150),
                                        Textarea::make('description')
                                            ->label('Meta Description')
                                            ->rows(3),
                                    ])
                            ]),
                    ])
                    ->columnSpanFull()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('classified_category_name')
                    ->label('Category')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('parent.classified_category_name')
                    ->label('Parent')
                    ->searchable()
                    ->sortable(false),
                TextColumn::make('slug')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('classified_category_description')
                    ->label('Description')
                    ->html()
                    ->limit(50)
                    ->searchable(),
                TextColumn::make('classifieds_count')
                    ->label('Count')
                    ->sortable(),
                ToggleColumn::make('is_active')
                    ->label('Visible'),
            ])
            ->defaultSort('created_at', 'desc')
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('classified_categories.*')
                ->selectSub(
                    function ($subQuery) {
                        $subQuery->from('classified_classified_category')
                                 ->selectRaw('COUNT(*)')
                                 ->whereColumn('classified_classified_category.classified_category_id', 'classified_categories.id');
                    },
                    'classifieds_count'
                );
                return $query;
            })
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\Filter::make('is_active')
                    ->label('Is Visible')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (ClassifiedCategory $record): string => route('classified.category', ['slug' => $record->slug]))
                        ->hidden(fn(ClassifiedCategory $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Tables\Actions\EditAction::make()
                        ->slideOver()   
                        ->modalWidth('3xl')
                        ->hidden(fn(ClassifiedCategory $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->excludeAttributes(['classifieds_count'])
                        ->hidden(fn(ClassifiedCategory $record): bool => $record->trashed()),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(ClassifiedCategory $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),
                    
                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageClassifiedCategories::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
