<?php

namespace App\Filament\Resources\Classified;

use App\Filament\Resources\Classified\ClassifiedDeliveryResource\Pages;
use App\Filament\Resources\Classified\ClassifiedDeliveryResource\RelationManagers;
use App\Models\Classified\ClassifiedDelivery;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Illuminate\Support\Str;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Set;
use Filament\Forms\Get;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Forms\Components\Section;
use App\Filament\Traits\HasSeoSection;

class ClassifiedDeliveryResource extends Resource
{
    use HasSeoSection;
    protected static ?string $model = ClassifiedDelivery::class;

    protected static ?string $navigationIcon = 'heroicon-o-truck';
    protected static ?string $navigationLabel = 'Deliveries';
    protected static ?string $navigationGroup = 'Classified';
    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Delivery')
                    ->tabs([
                        Tabs\Tab::make('Delivery')
                            ->schema([
                                TextInput::make('name')
                                    ->label('Delivery Name')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                        if ($context === 'create' && empty($get('slug'))) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (?string $state, Set $set) {
                                        if ($state) {
                                            $set('slug', Str::slug($state));
                                        }
                                    })
                                    ->dehydrated()
                                    ->unique(ignoreRecord: true),
                                Select::make('parent_id')
                                    ->label('Parent Delivery')
                                    ->relationship(name: 'parent', titleAttribute: 'name')
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent delivery')
                                    ->nullable()
                                    ->columnSpanFull(),
                                RichEditor::make('description')
                                    ->label('Description')
                                    ->disableToolbarButtons([
                                        'codeBlock',
                                    ])
                                    ->columnSpanFull(),
                                Toggle::make('is_active')
                                    ->label('Visible')
                                    ->default(true),
                            ]),
                        Tabs\Tab::make('SEO')
                            ->schema([
                               static::getSeoSection(),
                            ]),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('slug'),
                TextColumn::make('parent_id')
                    ->label('Parent')
                    ->formatStateUsing(fn($record) => $record->parent ? $record->parent->name : 'N/A')
                    ->sortable(),
                IconColumn::make('is_active')
                    ->boolean(),
                TextColumn::make('created_by')
                    ->label('Author')
                    ->formatStateUsing(fn($record) => $record->author?->name ?? 'N/A')
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\Filter::make('is_active')
                    ->label('Is Visible')
                    ->toggle()
                    ->query(fn(Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                ActionGroup::make([
                    EditAction::make()
                        ->slideOver()
                        ->modalWidth('3xl')
                        ->hidden(fn(ClassifiedDelivery $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->hidden(fn(ClassifiedDelivery $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(ClassifiedDelivery $record): bool => $record->trashed()),
                    ForceDeleteAction::make()
                        ->hidden(fn(ClassifiedDelivery $record): bool => $record->trashed()),
                    RestoreAction::make()
                        ->hidden(fn(ClassifiedDelivery $record): bool => !$record->trashed()),
                ])
                    ->label('Actions')
                    ->icon('heroicon-o-ellipsis-vertical')
                    ->color('gray'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageClassifiedDeliveries::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
