<?php

namespace App\Filament\Resources\Classified\ClassifiedResource\Pages;

use App\Filament\Resources\Classified\ClassifiedResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Location\City;
use App\Models\User\User;
use App\Enums\ClassifiedStatus;
use Filament\Notifications\Notification;
use Illuminate\View\View;

class CreateClassified extends CreateRecord
{
    protected static string $resource = ClassifiedResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'Save',
                    ClassifiedStatus::Scheduled->value => 'Schedule',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'heroicon-o-document-text',
                    ClassifiedStatus::Scheduled->value => 'heroicon-o-clock',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'primary',
                    ClassifiedStatus::Scheduled->value => 'warning',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();

                    if ($data['classified_status'] === ClassifiedStatus::Draft->value || $data['classified_status'] === ClassifiedStatus::Published->value) {
                        $this->data['classified_status'] = ClassifiedStatus::Published->value;
                        $this->data['published_at'] = now();
                    } elseif ($data['classified_status'] === ClassifiedStatus::Scheduled->value && empty($data['published_at'])) {
                        $this->data['published_at'] = now()->addHour();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['post_status'] = ClassifiedStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn () => ClassifiedResource::getUrl()),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['classified_status'] ?? null) {
                    ClassifiedStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    if ($this->data['classified_status'] === ClassifiedStatus::Draft->value || $this->data['classified_status'] === ClassifiedStatus::Published->value) {
                        $this->data['classified_status'] = ClassifiedStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),
            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['classified_status'] = ClassifiedStatus::Draft->value;
                    $this->create();
                }),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        if ($data['classified_status'] === ClassifiedStatus::Published->value) {
            $data['published_at'] = now();
        } elseif ($data['classified_status'] === ClassifiedStatus::Scheduled->value && empty($data['published_at'])) {
            $data['published_at'] = now()->addHour();
        }

        return $data;
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }

    protected function getCreatedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match($this->data['classified_status']) {
                ClassifiedStatus::Draft->value => 'Draft Saved',
                ClassifiedStatus::Pending->value => 'Classified Saved',
                ClassifiedStatus::Scheduled->value => 'Classified Scheduled',
                default => 'Classified Published',
            })
            ->body(match($this->data['classified_status']) {
                ClassifiedStatus::Draft->value => 'Your classified has been saved as draft.',
                ClassifiedStatus::Pending->value => 'Your classified has been saved as pending.',
                ClassifiedStatus::Scheduled->value => 'Your classified has been scheduled for publication.',
                default => 'Your classified has been published successfully.',
            });
    }

    protected function afterCreate(): void
    {
        $country = Country::firstOrCreate(
            ['country_code' => $this->data['country_code']],
            [
                'country_name' => $this->data['country_name'],
                'country_code' => $this->data['country_code'],
                'created_by' => auth()->user()->id
            ]
        );

        if ($this->data['state_name']) {
            $state = State::firstOrCreate(
                [
                    'country_id' => $country->id,
                    'state_name' => $this->data['state_name']
                ],
                [
                    'created_by' => auth()->user()->id
                ]
            );
            
            $this->record->state_id = $state->id;

            if ($this->data['city_name']) {
                $city = City::firstOrCreate(
                    [
                        'country_id' => $country->id,
                        'state_id' => $state->id,
                        'city_name' => $this->data['city_name']
                    ],
                    [
                        'created_by' => auth()->user()->id
                    ]
                );
                
                $this->record->city_id = $city->id;
            }
        }

        if ($this->data['country_name']) {
            $this->record->country_id = $country->id;
        }

        $this->record->save();
    }
}
