<?php

namespace App\Filament\Resources\Classified\ClassifiedResource\Pages;

use App\Filament\Resources\Classified\ClassifiedResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Cheesegrits\FilamentGoogleMaps\Concerns\InteractsWithMaps;
use App\Models\Location\Country;
use App\Enums\ClassifiedStatus;
use App\Models\Classified\Classified;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use Filament\Forms\Components\TextInput;
use Illuminate\View\View;

class EditClassified extends EditRecord
{
    use InteractsWithMaps;

    protected static string $resource = ClassifiedResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['classified_status']) {
                ClassifiedStatus::Draft->value => 'Draft Updated',
                ClassifiedStatus::Pending->value => 'Classified Updated',
                ClassifiedStatus::Scheduled->value => 'Schedule Updated',
                default => 'Classified Updated',
            })
            ->body(match ($this->data['classified_status']) {
                ClassifiedStatus::Draft->value => 'Your classified has been updated and saved as draft.',
                ClassifiedStatus::Pending->value => 'Your classified has been updated and saved as pending.',
                ClassifiedStatus::Scheduled->value => 'Your classified schedule has been updated.',
                default => 'Your classified has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),

            Actions\Action::make('preview')
                ->label('Preview')
                ->icon('heroicon-o-eye')
                ->color('success')
                ->url(fn() => route('classified.show', [
                    'slug' => $this->record->slug,
                    'preview' => $this->record->classified_status !== ClassifiedStatus::Published ? true : null,
                ]))
                ->openUrlInNewTab(),

            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of actions
                    Actions\Action::make('history')
                        ->label('View History')
                        ->icon('heroicon-o-clock')
                        ->color('info')
                        ->action(fn(Classified $record) => $record)
                        ->modalContent(view('filament.resources.includes.history', ['record' => $this->record]))
                        ->modalWidth('3xl')
                        ->modalHeading(fn(Classified $record) => 'History for ' . $record->title)
                        ->modalSubmitAction(false),

                    Actions\DeleteAction::make()
                        ->label('Move to Trash')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->hidden(fn(): bool => $this->record->trashed())
                        ->modalHeading('Delete Post')
                        ->modalDescription('Are you sure you want to delete this post? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, delete it')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post deleted')
                                ->body('The post has been moved to trash.')
                        ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn() => ClassifiedResource::getUrl()),
            ])
                ->tooltip('More Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('info')
                ->button()
                ->extraAttributes([
                    'class' => 'more-actions-btn',
                ]),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),

            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['classified_status' => ClassifiedStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['classified_status' => ClassifiedStatus::Draft]);
    }

    public function afterSave(): void
    {
        if ($this->record->classified_status !== ClassifiedStatus::Trashed && $this->record->trashed()) {
            $this->record->restore();
        }

        $country = Country::firstOrCreate(
            ['country_code' => $this->data['country_code']],
            [
                'country_name' => $this->data['country_name'],
                'country_code' => $this->data['country_code'],
                'created_by' => auth()->id()
            ]
        );

        if ($this->data['state_name']) {
            $state = \App\Models\Location\State::firstOrCreate(
                [
                    'country_id' => $country->id,
                    'state_name' => $this->data['state_name']
                ],
                [
                    'created_by' => auth()->id()
                ]
            );

            $this->record->state_id = $state->id;

            if ($this->data['city_name']) {
                $city = \App\Models\Location\City::firstOrCreate(
                    [
                        'country_id' => $country->id,
                        'state_id' => $state->id,
                        'city_name' => $this->data['city_name']
                    ],
                    [
                        'created_by' => auth()->id()
                    ]
                );

                $this->record->city_id = $city->id;
            }
        }

        if ($this->data['country_name']) {
            $this->record->country_id = $country->id;
        }

        $this->record->save();
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        if (isset($data['country_id'])) {
            $country = Country::find($data['country_id']);
            if ($country) {
                $data['country_name'] = $country->country_name;
                $data['country_code'] = $country->country_code;
            }
        }

        if (isset($data['state_id'])) {
            $state = \App\Models\Location\State::find($data['state_id']);
            if ($state) {
                $data['state_name'] = $state->state_name;
            }
        }

        if (isset($data['city_id'])) {
            $city = \App\Models\Location\City::find($data['city_id']);
            if ($city) {
                $data['city_name'] = $city->city_name;
            }
        }

        return $data;
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
