<?php

namespace App\Filament\Resources\Classified\ClassifiedResource\Pages;

use App\Filament\Resources\Classified\ClassifiedResource;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Filament\Resources\Components\Tab;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Cache;
use App\Enums\ClassifiedStatus;
use App\Models\Classified\Classified;


class ListClassifieds extends ListRecords
{
    protected static string $resource = ClassifiedResource::class;

    protected static string $view = 'vendor.filament.components.table.table-body';
    
    protected function getHeaderActions(): array
    {
        return [
            Actions\CreateAction::make()
                ->label('Add New')
                ->icon('heroicon-o-plus'),
        ];
    }

    public function getTabs(): array
    {
        // Get cached counts
        $cachedCounts = Cache::remember('classified-status-counts', 300, function () {
            $counts = Classified::query()
                ->select('classified_status')
                ->selectRaw('COUNT(*) as count')
                ->groupBy('classified_status')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->classified_status->value => $item->count];
                })
                ->toArray();

            $trashedCount = Classified::onlyTrashed()->count();
            $allCount = array_sum($counts) + $trashedCount;

            return [
                'counts' => $counts,
                'trashedCount' => $trashedCount,
                'allCount' => $allCount,
            ];
        });

        // Build tabs using the cached counts
        return [
            'all' => Tab::make('All')
                ->icon('heroicon-m-squares-2x2')
                ->badge($cachedCounts['allCount'])
                ->modifyQueryUsing(fn (Builder $query) => $query),
            'published' => Tab::make('Published')
                ->icon('heroicon-m-check-circle')
                ->badge($cachedCounts['counts'][ClassifiedStatus::Published->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('classified_status', ClassifiedStatus::Published)),
            'pending' => Tab::make('Pending')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][ClassifiedStatus::Pending->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('classified_status', ClassifiedStatus::Pending)),
            'draft' => Tab::make('Draft')
                ->icon('heroicon-m-document')
                ->badge($cachedCounts['counts'][ClassifiedStatus::Draft->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('classified_status', ClassifiedStatus::Draft)),
            'expired' => Tab::make('Expired')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][ClassifiedStatus::Expired->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('classified_status', ClassifiedStatus::Expired)),
            'trash' => Tab::make('Trash')
                ->icon('heroicon-m-trash')
                ->badge($cachedCounts['trashedCount'])
                ->modifyQueryUsing(fn (Builder $query) => $query->withTrashed()->whereNotNull('deleted_at')),
        ];
    }

    public function getDefaultActiveTab(): string | int | null
    {
        return 'all';
    }

    protected function getTableRecordsPerPageSelectOptions(): array
    {
        return [10, 25, 50];
    }

    public function getTableQuery(): Builder
    {
        return parent::getTableQuery()
            ->with(['media', 'author', 'categories'])
            ->latest();
    }   
}
