<?php

namespace App\Filament\Resources\Classified\ClassifiedResource\Traits;

use App\Enums\ClassifiedCondition;
use App\Enums\ClassifiedLabel;
use App\Enums\ClassifiedStatus;
use App\Filament\Traits\HasSeoSection;
use App\Models\User\User;
use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Str;

trait HasClassifiedForm
{
    use HasSeoSection;

    public function getClassifiedFormSchema($context = 'create', $panel = 'dashboard'): array
    {
        return [
            Grid::make()
                ->columns([
                    'default' => 1,
                    'lg'      => 10,
                ])
                ->schema([
                    // Main Content Column (70%)
                    Group::make()
                        ->columnSpan(['lg' => 7])
                        ->schema([
                            Section::make('Classified Details')
                                ->schema([
                                    TextInput::make('title')
                                        ->label('Title')
                                        ->placeholder('Enter the classified title')
                                        ->required()
                                        ->maxLength(191)
                                        ->autofocus()
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) use ($context) {
                                            if ($context === 'create' && empty($get('slug'))) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->validationMessages([
                                            'regex' => 'Title must only contain letters, numbers, spaces, and &, .,',
                                        ]),
                                    TextInput::make('slug')
                                        ->label('Slug')
                                        ->placeholder('Enter the post slug')
                                        ->helperText('Will be automatically generated from title if left empty.')
                                        ->maxLength(255)
                                        ->required()
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (string $state, Set $set) {
                                            $set('slug', Str::slug($state));
                                        })
                                        ->dehydrated(),
                                    RichEditor::make('description')
                                        ->disableToolbarButtons(['codeBlock'])
                                        ->required()
                                        ->columnSpanFull(),
                                ]),
                            Section::make('Classified Location')
                                ->schema([
                                    Grid::make()
                                        ->columns(2)
                                        ->schema([
                                            TextInput::make('location')
                                                ->columnSpanFull()
                                                ->required(),
                                            // First Column
                                            Group::make()
                                                ->columnSpan(1)
                                                ->schema([
                                                    Map::make('map_address')
                                                        ->mapControls([
                                                            'mapTypeControl'    => true,
                                                            'scaleControl'      => true,
                                                            'streetViewControl' => true,
                                                            'rotateControl'     => true,
                                                            'fullscreenControl' => true,
                                                            'searchBoxControl'  => false,
                                                            'zoomControl'       => true,
                                                        ])
                                                        ->defaultZoom(13)
                                                        ->defaultLocation([0, 0])
                                                        ->draggable()
                                                        ->clickable()
                                                        ->debug()
                                                        ->autocomplete('location')
                                                        ->autocompleteReverse()
                                                        ->reverseGeocode([
                                                            'address'      => '%n %S %D %L %z',
                                                            'postal_code'  => '%z',
                                                            'city_name'    => '%L',
                                                            'state_name'   => '%A1',
                                                            'state_code'   => '%a1',
                                                            'country_name' => '%C',
                                                            'country_code' => '%c',
                                                        ])
                                                        ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                                            if (isset($state['lat']) && isset($state['lng'])) {
                                                                $set('latitude', $state['lat']);
                                                                $set('longitude', $state['lng']);

                                                                if (isset($state['address'])) {
                                                                    $set('address', $state['address']);
                                                                }
                                                                if (isset($state['postal_code'])) {
                                                                    $set('postal_code', $state['postal_code']);
                                                                }
                                                                if (isset($state['country_name'])) {
                                                                    $set('country_name', $state['country_name']);
                                                                }
                                                                if (isset($state['country_code'])) {
                                                                    $set('country_code', $state['country_code']);
                                                                }
                                                                if (isset($state['state_name'])) {
                                                                    $set('state_name', $state['state_name']);
                                                                }
                                                                if (isset($state['state_code'])) {
                                                                    $set('state_code', $state['state_code']);
                                                                }
                                                                if (isset($state['city_name'])) {
                                                                    $set('city_name', $state['city_name']);
                                                                }
                                                            }
                                                        })
                                                        ->afterStateHydrated(function ($component, $state, $record) {
                                                            if ($record && $record->latitude && $record->longitude) {
                                                                $component->state([
                                                                    'lat'     => (float) $record->latitude,
                                                                    'lng'     => (float) $record->longitude,
                                                                    'address' => $record->map_address ?: $record->address,
                                                                ]);
                                                            }
                                                        }),
                                                ]),
                                            // Second Column
                                            Group::make()
                                                ->columnSpan(1)
                                                ->schema([
                                                    Grid::make()
                                                        ->columns(2)
                                                        ->schema([
                                                            TextInput::make('address')
                                                                ->label('Street Address')
                                                                ->required()
                                                                ->columnSpanFull(),
                                                            TextInput::make('latitude')
                                                                ->numeric()
                                                                ->columnSpan(1),
                                                            TextInput::make('longitude')
                                                                ->numeric()
                                                                ->columnSpan(1),
                                                            TextInput::make('country_name')
                                                                ->label('Country')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('state_name')
                                                                ->label('State')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('city_name')
                                                                ->label('City')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('postal_code')
                                                                ->label('Postal Code'),
                                                            Hidden::make('state_code'),
                                                            Hidden::make('country_code'),
                                                            Hidden::make('country_id'),
                                                            Hidden::make('state_id'),
                                                            Hidden::make('city_id'),
                                                        ]),
                                                ]),
                                        ]),
                                ]),
                            Section::make('Gallery')
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('gallery')
                                        ->label('Gallery')
                                        ->extraAttributes([
                                            'class' => 'upload-media-gallery',
                                        ])
                                        ->collection('gallery')
                                        ->multiple()
                                        ->image()
                                        ->reorderable()
                                        ->columnSpanFull()
                                        ->maxSize(5120)
                                        ->when($panel === 'admin', fn($component) => $component->imageEditor()->image()),
                                ]),
                            Section::make('Key Features')
                                ->schema([
                                    Repeater::make('key_features')
                                        ->label('Features')
                                        ->schema([
                                            Grid::make()
                                                ->columns(2)
                                                ->schema([
                                                    TextInput::make('key')
                                                        ->label('Feature Name')
                                                        ->placeholder('Enter feature title here e.g. request financing info, contact seller, etc.')
                                                        ->maxLength(20)
                                                        ->required(),
                                                    TextInput::make('value')
                                                        ->label('Feature Value')
                                                        ->placeholder('Enter feature value here e.g. financing info, seller contact, etc.')
                                                        ->required()
                                                        ->maxLength(100),
                                                ]),
                                        ]),
                                ]),
                            Section::make('Maintenance Detail')
                                ->schema([
                                    RichEditor::make('maintenance_detail')
                                        ->toolbarButtons([
                                            'bold',
                                            'italic',
                                            'underline',
                                            'bulletList',
                                            'orderedList',
                                            'undo',
                                            'redo',
                                            'underline',
                                        ])
                                        ->columnSpanFull(),
                                    SpatieMediaLibraryFileUpload::make('maintenance_docs')
                                        ->extraAttributes([
                                            'class' => 'upload-maintenance-docs',
                                        ])
                                        ->reorderable()
                                        ->collection('maintenance_docs')
                                        ->maxFiles(5)
                                        ->multiple()
                                        ->acceptedFileTypes([
                                            'application/pdf',
                                            'image/jpeg',
                                            'image/png',
                                            'application/msword',
                                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                                        ])
                                        ->columnSpanFull(),
                                ]),
                            // SEO Section
                            static::getSeoSection(),
                        ]),
                    // Sidebar Column (30%)
                    Group::make()
                        ->columnSpan(['lg' => 3])
                        ->schema([
                            Section::make('Settings')
                                ->schema(array_filter([
                                    // Admin-only fields
                                    $panel === 'admin' ? Select::make('classified_status')
                                        ->options([
                                            ClassifiedStatus::Draft->value     => ClassifiedStatus::Draft->name,
                                            ClassifiedStatus::Published->value => ClassifiedStatus::Published->name,
                                            ClassifiedStatus::Pending->value   => ClassifiedStatus::Pending->name,
                                            ClassifiedStatus::Expired->value   => ClassifiedStatus::Expired->name,
                                        ])
                                        ->label('Status')
                                        ->live()
                                        ->default(ClassifiedStatus::Draft->value)
                                        ->afterStateUpdated(function ($state, Set $set) {
                                            if ($state === ClassifiedStatus::Scheduled->value) {
                                                $set('published_at', now()->addHour());
                                            }
                                        }) : null,
                                    $panel === 'admin' ? DateTimePicker::make('published_at')
                                        ->label('Published Date')
                                        ->seconds(false)
                                        ->visible(fn(Get $get): bool => in_array($get('classified_status'), [
                                            ClassifiedStatus::Scheduled->value,
                                        ]))
                                        ->required(fn(Get $get): bool => $get('classified_status') === ClassifiedStatus::Scheduled->value)
                                        ->minDate(fn(Get $get): ?\DateTime => $get('classified_status') === ClassifiedStatus::Scheduled->value ? now() : null)
                                        ->helperText(fn(Get $get): string => $get('classified_status') === ClassifiedStatus::Scheduled->value
                                            ? 'Select when this classified should be published'
                                            : 'The date when this classified was published') : null,
                                    $panel === 'admin' ? Select::make('user_id')
                                        ->options(User::all()->pluck('name', 'id'))
                                        ->label('User')
                                        ->preload()
                                        ->required()
                                        ->default(fn() => Auth::check() ? Auth::id() : null)
                                        ->searchable() : null,
                                    // Dashboard-only fields
                                    $panel === 'dashboard' ? Hidden::make('post_status')
                                        ->default(ClassifiedStatus::Pending->value) : null,
                                    // Common fields
                                    SpatieMediaLibraryFileUpload::make('featured_image')
                                        ->collection('featured_image')
                                        ->imageEditor()
                                        ->required()
                                        ->conversion('thumb'),
                                    $panel === 'admin' ? Toggle::make('is_featured')
                                        ->label('Set as Featured Post') : null,
                                    TextInput::make('price')
                                        ->required()
                                        ->numeric()
                                        ->inputMode('decimal')
                                        ->default(0.00)
                                        ->prefix(fn() => config('app.currency_symbol', '$'))
                                        ->rules([
                                            'min:0.01',
                                            fn($get) => function($attribute, $value, $fail) use ($get) {
                                                $discountPrice = $get('discount_price');
                                                if ($discountPrice && $value <= $discountPrice) {
                                                    $fail("The price must be greater than the discount price.");
                                                }
                                            }
                                        ])
                                        ->validationMessages([
                                            'min' => 'The price must be greater than 0.',
                                        ]),
                                    TextInput::make('discount_price')
                                        ->numeric()
                                        ->inputMode('decimal')
                                        ->default(0.00)
                                        ->prefix(fn() => config('app.currency_symbol', '$'))
                                        ->rules([
                                            fn($get) => function($attribute, $value, $fail) use ($get) {
                                                $price = $get('price');
                                                if ($price && $value) {
                                                    if ($value >= $price) {
                                                        $fail("The discount price must be less than the regular price.");
                                                    }
                                                }
                                            }
                                        ]),

                                    DateTimePicker::make('discount_start')
                                        ->label('Discount Start')
                                        ->live(),
                                    DateTimePicker::make('discount_end')
                                        ->label('Discount End')
                                        ->rules([
                                            fn($get) => function($attribute, $value, $fail) use ($get) {
                                                $startDate = $get('discount_start');
                                                if ($startDate && $value && $value <= $startDate) {
                                                    $fail("The discount end date must be after the start date.");
                                                }
                                            }
                                        ]),
                                    Select::make('condition')
                                        ->label('Condition')
                                        ->options(ClassifiedCondition::class)
                                        ->default(ClassifiedCondition::Used->value),
                                    DatePicker::make('manufacture_date')
                                        ->label('Manufacture Date'),
                                    Select::make('label')
                                        ->label('Label')
                                        ->options(ClassifiedLabel::class)
                                        ->default(ClassifiedLabel::ForSale->value),
                                    TextInput::make('website_link')
                                        ->label('Website Link')
                                        ->url(),
                                    // Categories/Types/Tags - different relationship names between admin and dashboard
                                    $panel === 'admin' ? Select::make('classified_categories')
                                        ->multiple()
                                        ->relationship(
                                            name: 'Categories',
                                            titleAttribute: 'classified_category_name',
                                            modifyQueryUsing: fn(Builder $query) => $query
                                                ->select(['classified_categories.id', 'classified_categories.classified_category_name'])
                                        )
                                        ->createOptionForm([
                                            TextInput::make('classified_category_name')
                                                ->required()
                                                ->label('Category Name')
                                                ->maxLength(255)
                                                ->unique('classified_categories', 'classified_category_name'),
                                        ])
                                        ->createOptionAction(
                                            fn(FormAction $action) => $action
                                                ->modalWidth('sm')
                                                ->modalHeading('Create Category')
                                        )
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading categories...')
                                        ->noSearchResultsMessage('No categories found.')
                                        ->placeholder('Select categories') : Select::make('categories')
                                        ->multiple()
                                        ->placeholder('Select categories')
                                        ->relationship('categories', 'classified_category_name')
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading categories...')
                                        ->noSearchResultsMessage('No categories found.')
                                        ->placeholder('Select categories'),
                                    $panel === 'admin' ? Select::make('classified_types')
                                        ->multiple()
                                        ->relationship(
                                            name: 'Types',
                                            titleAttribute: 'classified_type_name',
                                            modifyQueryUsing: fn(Builder $query) => $query
                                                ->select(['classified_types.id', 'classified_types.classified_type_name'])
                                        )
                                        ->createOptionForm([
                                            TextInput::make('classified_type_name')
                                                ->required()
                                                ->label('Type Name')
                                                ->maxLength(255)
                                                ->unique('classified_types', 'classified_type_name'),
                                        ])
                                        ->createOptionAction(
                                            fn(FormAction $action) => $action
                                                ->modalWidth('sm')
                                                ->modalHeading('Create Type')
                                        )
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading types...')
                                        ->noSearchResultsMessage('No types found.')
                                        ->placeholder('Select types') : Select::make('types')
                                        ->multiple()
                                        ->placeholder('Select types')
                                        ->relationship('types', 'classified_type_name')
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading types...')
                                        ->noSearchResultsMessage('No types found.')
                                        ->placeholder('Select types'),
                                    $panel === 'admin' ? Select::make('classified_tags')
                                        ->multiple()
                                        ->relationship(
                                            name: 'Tags',
                                            titleAttribute: 'classified_tag_name',
                                            modifyQueryUsing: fn(Builder $query) => $query
                                                ->select(['classified_tags.id', 'classified_tags.classified_tag_name'])
                                        )
                                        ->createOptionForm([
                                            TextInput::make('classified_tag_name')
                                                ->required()
                                                ->label('Tag Name')
                                                ->maxLength(255)
                                                ->unique('classified_tags', 'classified_tag_name'),
                                        ])
                                        ->createOptionAction(
                                            fn(FormAction $action) => $action
                                                ->modalWidth('sm')
                                                ->modalHeading('Create Tag')
                                        )
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading tags...')
                                        ->noSearchResultsMessage('No tags found.')
                                        ->placeholder('Select tags') : Select::make('tags')
                                        ->multiple()
                                        ->placeholder('Select tags')
                                        ->relationship('tags', 'classified_tag_name')
                                        ->preload()
                                        ->searchable()
                                        ->optionsLimit(100)
                                        ->loadingMessage('Loading tags...')
                                        ->noSearchResultsMessage('No tags found.')
                                        ->placeholder('Select tags'),
                                    // Classified Delivery
                                    Select::make('deliveries')
                                        ->label('Classified Delivery')
                                        ->relationship('deliveries', 'name')
                                        ->multiple()
                                        ->placeholder('Select deliveries'),
                                ])),
                            Section::make('Video')
                                ->visible(function () use ($panel) {
                                    // Admin panel: always show Video section
                                    if ($panel === 'admin') {
                                        return true;
                                    }

                                    // Dashboard panel: show only if user passes the 'isService' Gate
                                    if ($panel === 'dashboard') {
                                        $user = Auth::user();
                                        // return $user ? Gate::forUser($user)->allows('isService') : false;
                                        return true;
                                    }

                                    // Other panels: hide by default
                                    return false;
                                })
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('video')
                                        ->collection('video')
                                        ->acceptedFileTypes(['video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov', 'video/wmv', 'video/flv']),
                                ]),

                        ]),
                ]),
        ];
    }
}
