<?php

namespace App\Filament\Resources\Classified\ClassifiedTagResource\Pages;

use App\Filament\Resources\Classified\ClassifiedTagResource;
use Filament\Resources\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Forms\Contracts\HasForms;
use Filament\Tables\Contracts\HasTable;
use App\Models\Classified\ClassifiedTag;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Table;
use Filament\Forms\Form;
use Illuminate\Support\Str;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Notifications\Notification;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Forms\Get;
use Filament\Forms\Set;

class ClassifiedTags extends Page implements HasForms, HasTable
{
    use InteractsWithForms, InteractsWithTable;

    protected static string $resource = ClassifiedTagResource::class;

    protected static string $view = 'filament.resources.classified.classified-tag-resource.pages.classified-tags';

    public ?array $data = [];
    public ?ClassifiedTag $classifiedTag = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('classified_tag_name')
                    ->label('Name')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->classifiedTag && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(100)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                Toggle::make('is_active')
                    ->label('Active')
                    ->default(true),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(ClassifiedTag::query())
            ->columns([
                TextColumn::make('classified_tag_name')
                    ->label('Name')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('classifieds_count')
                    ->label('Count')
                    ->sortable(),
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
                Filter::make('is_active')
                    ->label('Is Active')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (ClassifiedTag $record): string => route('classified.tag', ['slug' => $record->slug]))
                        ->hidden(fn(ClassifiedTag $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(ClassifiedTag $record): bool => $record->trashed())
                        ->action(function (ClassifiedTag $record): void {
                            $this->classifiedTag = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['classifieds_count'])
                        ->hidden(fn(ClassifiedTag $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(ClassifiedTag $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('classified_tags.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('classified_classified_tag')
                                    ->selectRaw('COUNT(*)')
                                    ->whereColumn('classified_classified_tag.classified_tag_id', 'classified_tags.id');
                        },
                        'classifieds_count'
                    );
                return $query;
            });
    }

    public function getTitle(): string
    {
        return 'Classified Tags';
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->classifiedTag) {
            // Update existing tag
            $this->classifiedTag->update($data);
            $message = 'Tag updated successfully';
        } else {
            // Create new tag
            ClassifiedTag::create($data);
            $message = 'Tag created successfully';
        }

        $this->classifiedTag = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->classifiedTag = null;
        $this->data = [];
        $this->form->fill();
    }
}
