<?php

namespace App\Filament\Resources\Classified;

use App\Filament\Resources\Classified\ClassifiedTypeResource\Pages;
use App\Models\Classified\ClassifiedType;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Textarea;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Section;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Select;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Str;

class ClassifiedTypeResource extends Resource
{
    protected static ?string $model = ClassifiedType::class;

    protected static ?string $navigationGroup = 'Classified';
    protected static ?string $navigationLabel = 'Types';
    protected static ?int $navigationSort = 4;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Type')
                    ->tabs([
                        Tabs\Tab::make('Type')
                            ->schema([
                                TextInput::make('classified_type_name')
                                    ->label('Type Name')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Set $set, ?string $state) {
                                        $set('slug', Str::slug($state));
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (?string $state, Set $set) {
                                        if ($state) {
                                            $set('slug', Str::slug($state));
                                        }
                                    })
                                    ->dehydrated()
                                    ->unique(ignoreRecord: true),
                                Select::make('parent_id')
                                    ->label('Parent Type')
                                    ->relationship(
                                        'parent',
                                        'classified_type_name',
                                        fn(Builder $query) => $query->where('is_active', true)
                                    )
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent type')
                                    ->nullable(),
                                RichEditor::make('classified_type_description')
                                    ->label('Description')
                                    ->disableToolbarButtons([
                                        'codeBlock',
                                    ])
                                    ->columnSpanFull(),
                                Toggle::make('is_active')
                                    ->label('Visible')
                                    ->default(true),
                            ]),
                        Tabs\Tab::make('SEO')
                            ->schema([
                                Section::make()
                                    ->relationship('seo')
                                    ->schema([
                                        TextInput::make('title')
                                            ->label('Meta Title')
                                            ->maxLength(150),
                                        Textarea::make('description')
                                            ->label('Meta Description')
                                            ->rows(3),
                                    ])
                            ]),
                    ])
                    ->columnSpanFull()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('classified_type_name')
                    ->label('Type')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('parent.classified_type_name')
                    ->label('Parent')
                    ->searchable()
                    ->sortable(false),
                TextColumn::make('slug')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('classified_type_description')
                    ->label('Description')
                    ->html()
                    ->limit(50)
                    ->searchable(),
                TextColumn::make('classifieds_count')
                    ->label('Count')
                    ->sortable(),
                ToggleColumn::make('is_active')
                    ->label('Visible'),
            ])
            ->defaultSort('created_at', 'desc')
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('classified_types.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('classified_classified_type')
                                ->selectRaw('COUNT(*)')
                                ->whereColumn('classified_classified_type.classified_type_id', 'classified_types.id');
                        },
                        'classifieds_count'
                    );
                return $query;
            })
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\Filter::make('is_active')
                    ->label('Is Visible')
                    ->toggle()
                    ->query(fn(Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make()
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        ->hidden(fn(ClassifiedType $record): bool => $record->trashed()),
                    Tables\Actions\EditAction::make()
                        ->icon('heroicon-o-pencil-square')
                        ->slideOver()
                        ->modalWidth('3xl')
                        ->hidden(fn(ClassifiedType $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->excludeAttributes(['classifieds_count'])
                        ->hidden(fn(ClassifiedType $record): bool => $record->trashed()),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(ClassifiedType $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),

                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageClassifiedTypes::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
