<?php

namespace App\Filament\Resources\Equipment;

use App\Filament\Resources\Equipment\EquipmentResource\Pages;
use App\Models\Equipment\Equipment;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use App\Enums\EquipmentStatus;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Illuminate\Support\Str;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Traits\HasSeoSection;



class EquipmentResource extends Resource
{
    use HasSeoSection;
    protected static ?string $model = Equipment::class;


    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10,
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7]) // Change to 7 for 70%
                            ->schema([
                                Section::make('Equipment Details')
                                    ->schema([
                                        TextInput::make('title')
                                            ->maxLength(191)
                                            ->required()
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Set $set, ?string $state) {
                                                $set('slug', Str::slug($state));
                                            }),
                                        TextInput::make('slug')
                                            ->maxLength(255)
                                            ->required()
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (?string $state, Set $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        RichEditor::make('description')
                                            ->disableToolbarButtons(['codeBlock',]),

                                    ]),
                                Section::make('Gallery Images')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('gallery')
                                            ->extraAttributes([
                                                'class' => 'upload-media-gallery',
                                            ])
                                            ->collection('gallery')
                                            ->label('Image')
                                            ->multiple(),
                                    ]),
                                Section::make('Equipmnent Content')
                                    ->schema([
                                        RichEditor::make('use_case')
                                            ->toolbarButtons(['bold', 'bulletList', 'italic', 'underline', 'orderedList', 'redo', 'undo']),
                                        RichEditor::make('constraint')
                                            ->toolbarButtons(['bold', 'bulletList', 'italic', 'underline', 'orderedList', 'redo', 'undo']),
                                    ]),
                                Section::make('Equipment Address')
                                    ->schema([
                                        TextInput::make('address')
                                            ->maxLength(191),
                                        Select::make('country_id')
                                            ->relationship('country', 'country_name')
                                            ->searchable()
                                            ->preload()
                                            ->live(),
                                        Select::make('state_id')
                                            ->relationship('state', 'state_name')
                                            ->searchable()
                                            ->preload()
                                            ->live()
                                            ->required(fn(Get $get) => filled($get('country_id')))
                                            ->options(
                                                fn(Get $get) =>
                                                \App\Models\Location\State::query()
                                                    ->where('country_id', $get('country_id'))
                                                    ->pluck('state_name', 'id')
                                            ),
                                        Select::make('city_id')
                                            ->relationship('city', 'city_name')
                                            ->searchable()
                                            ->preload()
                                            ->required(fn(Get $get) => filled($get('state_id')))
                                            ->options(
                                                fn(Get $get) =>
                                                \App\Models\Location\City::query()
                                                    ->where('state_id', $get('state_id'))
                                                    ->pluck('city_name', 'id')
                                            ),
                                    ]),
                                Section::make('Additional Resources')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('additional_resources')
                                            ->extraAttributes([
                                                'class' => 'upload-additional-resources',
                                            ])
                                            ->collection('additional_resources')
                                            ->maxFiles(5)
                                            ->multiple()
                                            ->acceptedFileTypes([
                                                'application/pdf',
                                                'image/jpeg',
                                                'image/png',
                                                'application/msword',
                                                'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
                                            ])
                                            ->columnSpanFull(),
                                    ]),

                                Section::make('Specifications')
                                    ->schema([
                                        Repeater::make('specification')
                                            ->schema([
                                                TextInput::make('specification_name')
                                                    ->label('Specification Name')
                                                    ->required()
                                                    ->maxLength(30)
                                                    ->columnSpan(['lg' => 6]),
                                                TextInput::make('specification_value')
                                                    ->label('Specification Value')
                                                    ->required()
                                                    ->maxLength(20)
                                                    ->columnSpan(['lg' => 6]),
                                            ])->columns(12)
                                            ->maxItems(10),
                                    ]),
                                static::getSeoSection(),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3]) // Change to 3 for 30%
                            ->schema([
                                Section::make('Settings')
                                    ->schema([
                                        Select::make('user_id')
                                            ->label('Belongs To')
                                            ->relationship('user', 'name')
                                            ->preload()
                                            ->required()
                                            ->searchable()
                                            ->helperText('Select the user that belongs to this equipment'),
                                        Select::make('categories')
                                            ->label('Category')
                                            ->maxItems(2)
                                            ->relationship(titleAttribute: 'category_name')
                                            ->preload()
                                            ->multiple()
                                            ->required()
                                            ->searchable()
                                            ->helperText('Select the category for the equipment'),
                                        Select::make('status')
                                            ->options(EquipmentStatus::class)
                                            ->default(EquipmentStatus::Draft->value)
                                            ->required(),
                                        TagsInput::make('tags')
                                            ->rules(['max:15'])
                                            ->label('Tags')
                                            ->placeholder('Enter the tags for the equipment')
                                            ->helperText('Type the tag and press enter to add'),
                                    ]),
                                Section::make('Featured Image')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('featured_image')
                                            ->collection('featured_image')
                                            ->label('Image')
                                            ->image(),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                Tables\Columns\TextColumn::make('title')
                    ->formatStateUsing(fn($state) => Str::limit($state, 30))
                    ->searchable(),
                Tables\Columns\TextColumn::make('status')->badge()
                    ->formatStateUsing(fn($state) => $state->getLabel())
                    ->sortable(),
                Tables\Columns\TextColumn::make('tags')
                    ->formatStateUsing(function ($state) {
                        return $state;
                    })
                    ->wrap()
                    ->badge()
                    ->listWithLineBreaks()
                    ->searchable()
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options(EquipmentStatus::class)
                    ->label('Status')
                    ->placeholder('Select Status'),
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make(),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListEquipment::route('/'),
            'create' => Pages\CreateEquipment::route('/create'),
            'edit' => Pages\EditEquipment::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
