<?php

namespace App\Filament\Resources\Equipment\EquipmentResource\Pages;

use App\Filament\Resources\Equipment\EquipmentResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Support\Facades\Auth;
use Filament\Notifications\Notification;
use Illuminate\Contracts\View\View;
use App\Enums\EquipmentStatus;

class CreateEquipment extends CreateRecord
{
    protected static string $resource = EquipmentResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn() => match ($this->data['status'] ?? null) {
                    EquipmentStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn() => match ($this->data['status'] ?? null) {
                    EquipmentStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn() => match ($this->data['status'] ?? null) {
                    EquipmentStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();

                    if ($data['status'] === EquipmentStatus::Draft->value || $data['status'] === EquipmentStatus::Published->value) {
                        $this->data['status'] = EquipmentStatus::Published->value;
                        $this->data['published_at'] = now();
                    }

                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['status'] = EquipmentStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn() => EquipmentResource::getUrl()),
        ];
    }
    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        $data['created_by'] = Auth::user()->id;
        return $data;
    }

    protected function getCreatedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['status']) {
                EquipmentStatus::Draft->value => 'Draft Saved',
                EquipmentStatus::Pending->value => 'Equipment Saved',
                default => 'Equipment Published',
            })
            ->body(match ($this->data['status']) {
                EquipmentStatus::Draft->value => 'Your equipment has been saved as draft.',
                EquipmentStatus::Pending->value => 'Your equipment has been saved as pending.',
                default => 'Your equipment has been published successfully.',
            });
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
