<?php

namespace App\Filament\Resources\Equipment\EquipmentResource\Pages;

use App\Filament\Resources\Equipment\EquipmentResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use App\Enums\EquipmentStatus;
use Illuminate\Contracts\View\View;

class EditEquipment extends EditRecord
{
    protected static string $resource = EquipmentResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['status']) {
                EquipmentStatus::Draft->value => 'Draft Updated',
                EquipmentStatus::Pending->value => 'Equipment Updated',
                default => 'Equipment Updated',
            })
            ->body(match ($this->data['status']) {
                EquipmentStatus::Draft->value => 'Your equipment has been updated and saved as draft.',
                EquipmentStatus::Pending->value => 'Your equipment has been updated and saved as pending.',
                default => 'Your equipment has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),



            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of action

                    Actions\DeleteAction::make()
                        ->label('Move to Trash')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->hidden(fn(): bool => $this->record->trashed())
                        ->modalHeading('Delete Project')
                        ->modalDescription('Are you sure you want to delete this project? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, delete it')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project deleted')
                                ->body('The project has been moved to trash.')
                        ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn() => EquipmentResource::getUrl()),
            ])
                ->tooltip('More Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('info')
                ->button()
                ->extraAttributes([
                    'class' => 'more-actions-btn',
                ]),
        ];
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['status' => EquipmentStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['status' => EquipmentStatus::Draft]);
    }

    protected function afterSave(): void
    {
        if ($this->record->status !== EquipmentStatus::Trashed && $this->record->trashed()) {
            $this->record->restore();
        }
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),

            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
