<?php

namespace App\Filament\Resources\Job;

use App\Filament\Resources\Job\JobApplicationResource\Pages;
use App\Filament\Resources\Job\JobApplicationResource\RelationManagers;
use App\Models\Job\JobApplication;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
class JobApplicationResource extends Resource
{
    protected static ?string $model = JobApplication::class;

    // protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'Job';
    protected static ?string $navigationLabel = 'Applications';
    protected static ?int $navigationSort = 4;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('job_id')
                    ->relationship('job', 'id')
                    ->required(),
                Forms\Components\Select::make('applicant_id')
                    ->relationship('applicant', 'name')
                    ->required(),
                Forms\Components\TextInput::make('application_status')
                    ->required()
                    ->maxLength(50)
                    ->default('applied'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('job.job_title')
                    ->formatStateUsing(fn ($state) => Str::limit($state, 50))
                    ->tooltip(fn ($state) => $state)
                    ->sortable(),
                Tables\Columns\TextColumn::make('applicant.name')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('application_status')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
                Tables\Actions\ForceDeleteAction::make(),
                Tables\Actions\RestoreAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageJobApplications::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
