<?php

namespace App\Filament\Resources\Job\JobCareerLevelResource\Pages;

use App\Filament\Resources\Job\JobCareerLevelResource;
use Filament\Resources\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use App\Models\Job\JobCareerLevel;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Str;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Filters\Filter;

class JobCareerLevels extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = JobCareerLevelResource::class;

    protected static string $view = 'filament.resources.job.job-career-level-resource.pages.job-career-level';

    public ?array $data = [];
    public ?JobCareerLevel $jobCareerLevel = null;

    public function mount(): void
    {   
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('career_level_name')
                    ->label('Title')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->jobCareerLevel && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (string $state, Set $set) {
                        $set('slug', Str::slug($state));
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                RichEditor::make('career_level_description')
                    ->disableToolbarButtons([
                        'codeBlock',
                    ])
                    ->label('Description')
                    ->columnSpanFull(),
                Toggle::make('is_active')
                    ->label('Active')
                    ->default(true),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(JobCareerLevel::query())
            ->columns([
                TextColumn::make('career_level_name')
                    ->label('Title')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                TextColumn::make('jobs_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
                Filter::make('is_active')
                    ->label('Is Active')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_active', true)),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (JobCareerLevel $record): string => route('job.career-level', ['slug' => $record->slug]))
                        ->hidden(fn(JobCareerLevel $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(JobCareerLevel $record): bool => $record->trashed())
                        ->action(function (JobCareerLevel $record): void {
                            $this->jobCareerLevel = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['jobs_count'])
                        ->hidden(fn(JobCareerLevel $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(JobCareerLevel $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                return $query->select('job_career_level.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('jobs')
                                ->whereColumn('jobs.job_career_level_id', 'job_career_level.id')
                                ->selectRaw('COUNT(*)');
                        },
                        'jobs_count'
                    );
            });
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->jobCareerLevel) {
            // Update existing tag
            $this->jobCareerLevel->update($data);
            $message = 'Job Career Level updated successfully';
        } else {
            // Create new tag
            JobCareerLevel::create($data);
            $message = 'Job Career Level created successfully';
        }

        $this->jobCareerLevel = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->jobCareerLevel = null;
        $this->data = [];
        $this->form->fill();
    } 
}


