<?php

namespace App\Filament\Resources\Job\JobIndustryResource\Pages;

use Filament\Resources\Pages\Page;
use Filament\Resources\Pages\Concerns\InteractsWithRecord;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\IconColumn;
use App\Models\Job\JobIndustry;
use Illuminate\Support\Str; 
use Filament\Forms\Form;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use App\Filament\Resources\Job\JobIndustryResource;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Forms\Get;
use Filament\Forms\Set;


class JobIndustries extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = JobIndustryResource::class;

    protected static string $view = 'filament.resources.job.job-industry-resource.pages.job-industries';

    public ?array $data = [];
    public ?JobIndustry $jobIndustry = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('industry_name')
                    ->label('Industry Name')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->jobIndustry && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                RichEditor::make('industry_description')
                    ->disableToolbarButtons([
                        'codeBlock',
                    ])
                    ->label('Description')
                    ->columnSpanFull(),
                Toggle::make('is_active')
                    ->label('Active')
                    ->default(true),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(JobIndustry::query())
            ->columns([
                TextColumn::make('industry_name')
                    ->label('Name')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                TextColumn::make('industries_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (JobIndustry $record): string => route('job.industry', ['slug' => $record->slug]))
                        ->hidden(fn(JobIndustry $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(JobIndustry $record): bool => $record->trashed())
                        ->action(function (JobIndustry $record): void {
                            $this->jobIndustry = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['industries_count'])
                        ->hidden(fn(JobIndustry $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(JobIndustry $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                return $query->select('job_industries.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('jobs')
                                ->whereColumn('jobs.job_industry_id', 'job_industries.id')
                                ->selectRaw('COUNT(*)');
                        },
                        'industries_count'
                    );
            });
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->jobIndustry) {
            // Update existing tag
            $this->jobIndustry->update($data);
            $message = 'Job Industry updated successfully';
        } else {
            // Create new tag
            JobIndustry::create($data);
            $message = 'Job Industry created successfully';
        }

        $this->jobIndustry = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->jobIndustry = null;
        $this->data = [];
        $this->form->fill();
    }
}
