<?php

namespace App\Filament\Resources\Job\JobResource\Pages;

use App\Filament\Resources\Job\JobResource;
use App\Models\Job\Job;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use Filament\Notifications\Notification;
use Illuminate\View\View;
use App\Enums\JobStatus;

class CreateJob extends CreateRecord
{
    protected static string $resource = JobResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'Save',
                    JobStatus::Scheduled->value => 'Schedule',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'heroicon-o-document-text',
                    JobStatus::Scheduled->value => 'heroicon-o-clock',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'primary',
                    JobStatus::Scheduled->value => 'warning',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();

                    if ($data['job_status'] === JobStatus::Draft->value || $data['job_status'] === JobStatus::Published->value) {
                        $this->data['job_status'] = JobStatus::Published->value;
                        $this->data['published_at'] = now();
                    } elseif ($data['job_status'] === JobStatus::Scheduled->value && empty($data['published_at'])) {
                        $this->data['published_at'] = now()->addHour();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['job_status'] = JobStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn () => JobResource::getUrl()),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['job_status'] ?? null) {
                    JobStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    if ($this->data['job_status'] === JobStatus::Draft->value || $this->data['job_status'] === JobStatus::Published->value) {
                        $this->data['job_status'] = JobStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),
            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['job_status'] = JobStatus::Draft->value;
                    $this->create();
                }),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        if ($data['job_status'] === JobStatus::Published->value) {
            $data['published_at'] = now();
        } elseif ($data['job_status'] === JobStatus::Scheduled->value && empty($data['published_at'])) {
            $data['published_at'] = now()->addHour();
        }

        return $data;
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
