<?php

namespace App\Filament\Resources\Job\JobResource\Pages;

use App\Filament\Resources\Job\JobResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use App\Enums\JobStatus;
use App\Models\Job\Job;
use Filament\Notifications\Notification;
use Illuminate\View\View;

class EditJob extends EditRecord
{
    protected static string $resource = JobResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match($this->data['job_status']) {
                JobStatus::Draft->value => 'Draft Updated',
                JobStatus::Pending->value => 'Job Updated',
                JobStatus::Scheduled->value => 'Schedule Updated',
                default => 'Job Updated',
            })
            ->body(match($this->data['job_status']) {
                JobStatus::Draft->value => 'Your job has been updated and saved as draft.',
                JobStatus::Pending->value => 'Your job has been updated and saved as pending.',
                JobStatus::Scheduled->value => 'Your job schedule has been updated.',
                default => 'Your job has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),

            Actions\Action::make('preview')
                ->label('Preview')
                ->icon('heroicon-o-eye')
                ->color('success')
                ->url(fn () => route('job.show', [
                    'slug' => $this->record->slug,
                    'preview' => $this->record->job_status !== JobStatus::Published ? true : null,
                ]))
                ->openUrlInNewTab(),

            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of actions
                Actions\Action::make('history')
                    ->label('View History')
                    ->icon('heroicon-o-clock')
                    ->color('info')
                    ->action(fn (Job $record) => $record)
                    ->modalContent(view('filament.resources.includes.history', ['record' => $this->record]))
                    ->modalWidth('3xl')
                    ->modalHeading(fn (Job $record) => 'History for ' . $record->title)
                    ->modalSubmitAction(false),

                Actions\DeleteAction::make()
                    ->label('Move to Trash')
                    ->icon('heroicon-o-trash')
                    ->requiresConfirmation()
                    ->hidden(fn (): bool => $this->record->trashed())
                    ->modalHeading('Delete Post')
                    ->modalDescription('Are you sure you want to delete this post? This action can be undone later.')
                    ->modalSubmitActionLabel('Yes, delete it')
                    ->successNotification(
                        Notification::make()
                            ->success()
                            ->title('Post deleted')
                            ->body('The post has been moved to trash.')
                    ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn () => JobResource::getUrl()),
            ])
            ->tooltip('More Actions')
            ->icon('heroicon-m-ellipsis-vertical')
            ->color('info')
            ->button()
            ->extraAttributes([
                'class' => 'more-actions-btn',
            ]),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),
            
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function afterSave(): void
    {
        $data = $this->data;
        if($this->record->job_status !== JobStatus::Trashed) {
            $this->record->restore();
        }
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['job_status' => JobStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['job_status' => JobStatus::Draft]);
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
