<?php

namespace App\Filament\Resources\Job\JobResource\Pages;

use App\Filament\Resources\Job\JobResource;
use App\Enums\JobStatus;
use Filament\Actions;
use App\Models\Job\Job;
use Filament\Resources\Components\Tab;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Builder;

class ListJobs extends ListRecords
{
    protected static string $resource = JobResource::class;

    protected static string $view = 'vendor.filament.components.table.table-body';

    protected function getHeaderActions(): array
    {
        return [
            Actions\CreateAction::make()
                ->icon('heroicon-o-plus')
                ->label('Add New')
                ->color('primary'),
        ];
    }

    public function getTabs(): array
    {
        // Get cached counts
        $cachedCounts = Cache::remember('job-status-counts', 300, function () {
            $counts = Job::query()
                ->select('job_status')
                ->selectRaw('COUNT(*) as count')
                ->groupBy('job_status')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->job_status->value => $item->count];
                })
                ->toArray();

            $trashedCount = Job::onlyTrashed()->count();
            $allCount = array_sum($counts) + $trashedCount;

            return [
                'counts' => $counts,
                'trashedCount' => $trashedCount,
                'allCount' => $allCount,
            ];
        });

        // Build tabs using the cached counts
        return [
            'all' => Tab::make()
                ->label('All')
                ->icon('heroicon-m-squares-2x2')
                ->badge($cachedCounts['allCount']),
            'published' => Tab::make()
                ->label('Published')
                ->icon('heroicon-m-check-circle')
                ->badge($cachedCounts['counts'][JobStatus::Published->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('job_status', JobStatus::Published)),
            'pending' => Tab::make()
                ->label('Pending Review')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][JobStatus::Pending->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('job_status', JobStatus::Pending)),
            'draft' => Tab::make()
                ->label('Draft')
                ->icon('heroicon-m-document')
                ->badge($cachedCounts['counts'][JobStatus::Draft->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('job_status', JobStatus::Draft)),
            'expired' => Tab::make()
                ->label('Expired')
                ->icon('heroicon-m-clock')
                ->badge($cachedCounts['counts'][JobStatus::Expired->value] ?? 0)
                ->modifyQueryUsing(fn (Builder $query) => $query->where('job_status', JobStatus::Expired)),
            'trash' => Tab::make()
                ->label('Trash')
                ->icon('heroicon-m-trash')
                ->badge($cachedCounts['trashedCount'])
                ->modifyQueryUsing(fn (Builder $query) => $query->withTrashed()->whereNotNull('deleted_at')),
        ];
    }

    public function getDefaultActiveTab(): string | int | null
    {
        return 'all';
    }

    protected function getTableRecordsPerPageSelectOptions(): array
    {
        return [10, 25, 50];
    }

    public function getTableQuery(): Builder
    {
        return parent::getTableQuery()
            ->with(['user', 'job_skills', 'job_career_level', 'job_type'])
            ->latest();
    }   
}
