<?php

namespace App\Filament\Resources\Job\JobResource\Traits;

use App\Enums\JobGender;
use App\Enums\JobShift;
use App\Enums\JobStatus;
use App\Enums\JobWorkMode;
use App\Enums\JobSalaryPeriod;
use App\Filament\Traits\HasSeoSection;
use App\Forms\Components\TinyMceEditor;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Tabs\Tab;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

trait HasJobForm
{
    use HasSeoSection;

    public function getJobFormSchema($context = 'create', $panel = 'dashboard'): array
    {
        return [
            Grid::make()
                ->columns([
                    'default' => 1,
                    'lg' => 10,
                ])
                ->schema([
                    // Main Content Column (70%)
                    Group::make()
                        ->columnSpan(['lg' => 7])
                        ->schema([
                            ///////// Basic Information
                            Section::make('Basic Information')
                                ->description('Enter the core details of the job posting')
                                ->schema([
                                    TextInput::make('job_title')
                                        ->label('Title')
                                        ->required()
                                        ->maxLength(191)
                                        ->placeholder('Enter the job title')
                                        ->autofocus()
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) use ($context) {
                                            if ($context === 'create' && empty($get('slug'))) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->validationMessages([
                                            'regex' => 'Title must only contain letters, numbers, spaces, and &.',
                                        ])
                                        ->columnSpanFull(),
                                    TextInput::make('slug')
                                        ->required()
                                        ->maxLength(191)
                                        ->helperText('Will be automatically generated from title if left empty.')
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (?string $state, Set $set) {
                                            if ($state) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->dehydrated()
                                        ->columnSpanFull(),
                                    TinyMceEditor::make('job_description')
                                        ->label('Job Description')
                                        ->required()
                                        ->plugins('lists link autolink autoresize')
                                        ->toolbar('bold italic | bullist numlist | link | h2 h3')
                                        ->menubar(false)
                                        ->branding(false)
                                        ->promotion(false)
                                        ->height(300)
                                        ->placeholder('Describe the job role and responsibilities')
                                        ->columnSpanFull(),
                                ])
                                ->collapsible(),

                            ///////// Job Specifications
                            Section::make('Job Specifications')
                                ->description('Define the key characteristics and requirements of the position')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            Select::make('job_role_id')
                                                ->relationship('job_role', 'job_role_name')
                                                ->required()
                                                ->searchable()
                                                ->preload()
                                                ->label('Role'),
                                            Select::make('job_type_id')
                                                ->relationship('job_type', 'job_type_name')
                                                ->required()
                                                ->searchable()
                                                ->preload()
                                                ->label('Employment Type'),
                                            Select::make('job_career_level_id')
                                                ->relationship('job_career_level', 'career_level_name')
                                                ->required()
                                                ->searchable()
                                                ->preload()
                                                ->label('Career Level'),
                                            Select::make('job_industry_id')
                                                ->relationship('job_industry', 'industry_name')
                                                ->required()
                                                ->searchable()
                                                ->preload()
                                                ->label('Industry'),
                                            Select::make('job_shift')
                                                ->options(JobShift::class)
                                                ->required()
                                                ->label('Work Shift'),
                                            Select::make('work_mode')
                                                ->options(JobWorkMode::class)
                                                ->required()
                                                ->label('Work Mode'),
                                            Select::make('preferred_gender')
                                                ->options(JobGender::class)
                                                ->nullable()
                                                ->label('Gender Preference'),
                                            TextInput::make('available_positions')
                                                ->numeric()
                                                ->default(1)
                                                ->required()
                                                ->minValue(1)
                                                ->label('Number of Positions'),
                                            Select::make('skills')
                                                ->relationship('job_skills', 'job_skill_name')
                                                ->multiple()
                                                ->preload()
                                                ->searchable()
                                                ->label('Required Skills')
                                                ->when($panel === 'admin', function ($component) {
                                                    return $component->createOptionForm([
                                                        TextInput::make('job_skill_name')
                                                            ->required()
                                                            ->maxLength(255)
                                                    ]);
                                                }),
                                        ]),
                                ])
                                ->collapsible(),

                            ///////// Requirements & Benefits
                            Section::make('Requirements & Benefits')
                                ->schema([
                                    Tabs::make('Requirements')
                                        ->tabs([
                                            Tab::make('Key Responsibilities')
                                                ->icon('heroicon-o-check-circle')
                                                ->schema([
                                                    TinyMceEditor::make('key_responsibilities')
                                                        ->plugins('lists autolink')
                                                        ->toolbar('bold | bullist')
                                                        ->menubar(false)
                                                        ->branding(false)
                                                        ->promotion(false)
                                                        ->height(250)
                                                        ->placeholder('List the main responsibilities')
                                                        ->columnSpanFull(),
                                                ]),
                                            Tab::make('Qualifications')
                                                ->icon('heroicon-o-academic-cap')
                                                ->schema([
                                                    TinyMceEditor::make('required_qualifications')
                                                        ->plugins('lists autolink')
                                                        ->toolbar('bold | bullist')
                                                        ->menubar(false)
                                                        ->branding(false)
                                                        ->promotion(false)
                                                        ->height(250)
                                                        ->placeholder('List required qualifications')
                                                        ->columnSpanFull(),
                                                ]),
                                            Tab::make('Benefits')
                                                ->icon('heroicon-o-gift')
                                                ->schema([
                                                    TinyMceEditor::make('bonus_and_benefits')
                                                        ->plugins('lists autolink')
                                                        ->toolbar('bold | bullist')
                                                        ->menubar(false)
                                                        ->branding(false)
                                                        ->promotion(false)
                                                        ->height(250)
                                                        ->placeholder('List the benefits and perks')
                                                        ->columnSpanFull(),
                                                ]),
                                            Tab::make('Nice to Have Skills')
                                                ->icon('heroicon-o-gift')
                                                ->schema([
                                                    TinyMceEditor::make('nice_to_have_skills')
                                                        ->plugins('lists autolink')
                                                        ->toolbar('bold | bullist')
                                                        ->menubar(false)
                                                        ->branding(false)
                                                        ->promotion(false)
                                                        ->height(250)
                                                        ->placeholder('List the benefits and perks')
                                                        ->columnSpanFull(),
                                                ]),
                                        ])
                                        ->columnSpanFull(),
                                ])
                                ->collapsible(),

                            // SEO Section
                            static::getSeoSection(),
                        ]),

                    // Sidebar Column (30%)
                    Group::make()
                        ->columnSpan(['lg' => 3])
                        ->schema([
                            Section::make('Settings')
                                ->schema(array_filter([
                                    // Admin-only fields
                                    $panel === 'admin' ? Select::make('job_status')
                                        ->options([
                                            JobStatus::Published->value => JobStatus::Published->name,
                                            JobStatus::Pending->value => JobStatus::Pending->name,
                                            JobStatus::Draft->value => JobStatus::Draft->name,
                                            JobStatus::Expired->value => JobStatus::Expired->name,
                                        ])
                                        ->label('Status')
                                        ->default(JobStatus::Draft->value)
                                        ->live()
                                        ->afterStateUpdated(function ($state, Set $set) {
                                            if ($state === JobStatus::Published->value) {
                                                $set('published_at', now());
                                            }
                                        }) : Hidden::make('job_status')
                                        ->default(JobStatus::Pending->value),

                                    $panel === 'admin' ? Select::make('user_id')
                                        ->relationship('user', 'name')
                                        ->label('Employer')
                                        ->preload()
                                        ->required()
                                        ->default(fn() => Auth::check() ? Auth::id() : null)
                                        ->searchable() : null,

                                    $panel === 'admin' ? DateTimePicker::make('published_at')
                                        ->label('Published Date')
                                        ->visible(fn(Get $get): bool => $get('job_status') === JobStatus::Published->value) : null,

                                    $panel === 'admin' ? Toggle::make('is_featured')
                                        ->label('Set as Featured Job') : null,

                                    Toggle::make('is_active')
                                        ->label('Active')
                                        ->default(true)
                                        ->inline(false)
                                        ->onIcon('heroicon-s-check')
                                        ->offIcon('heroicon-s-x-mark'),

                                    DatePicker::make('apply_by_date')
                                        ->label('Application Deadline')
                                        ->required()
                                        ->live(),


                                    $panel === 'admin'
                                        ? DatePicker::make('expired_at')
                                        ->label('Expiry Date')
                                        ->rule(fn(callable $get) => function ($attribute, $value, $fail) use ($get) {
                                            if ($value < $get('apply_by_date')) {
                                                $message = 'Expiry Date must be after the Application Deadline.';
                                                $fail($message);
                                                return $message;
                                            }
                                        })
                                        : DatePicker::make('expired_at')
                                        ->label('Expiry Date')

                                        ->rule(fn(callable $get) => function ($attribute, $value, $fail) use ($get) {
                                            if ($value < $get('apply_by_date')) {
                                                $fail('Expiry Date must be after the Application Deadline.');
                                            }
                                        }),
                                ]))
                                ->collapsible(),

                            // Compensation and Benefits
                            Section::make('Compensation')
                                ->schema([
                                    Select::make('currency_id')
                                        ->relationship('currency', 'currency_code')
                                        ->required()
                                        ->searchable()
                                        ->preload(),
                                    Grid::make(2)
                                        ->schema([
                                            TextInput::make('min_salary')
                                                ->numeric()
                                                ->placeholder('0')
                                                ->live(onBlur: true),
                                            TextInput::make('max_salary')
                                                ->numeric()
                                                ->placeholder('0')
                                                ->rules([
                                                    fn(Get $get) => function ($attribute, $value, $fail) use ($get) {
                                                        $minSalary = $get('min_salary');
                                                        if ($value && $minSalary && $value < $minSalary) {
                                                            $fail('Maximum salary cannot be less than minimum salary.');
                                                        }
                                                    }
                                                ]),
                                        ]),
                                    Select::make('salary_period')
                                        ->options(JobSalaryPeriod::class)
                                        ->required(),
                                ])
                                ->collapsible(),

                            // Location Section
                            Section::make('Location')
                                ->schema([
                                    Select::make('country_id')
                                        ->relationship('country', 'country_name')
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->live(),
                                    Select::make('state_id')
                                        ->relationship('state', 'state_name')
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->live()
                                        ->visible(fn(Get $get) => filled($get('country_id')))
                                        ->options(
                                            fn(Get $get) =>
                                            \App\Models\Location\State::query()
                                                ->where('country_id', $get('country_id'))
                                                ->pluck('state_name', 'id')
                                        ),
                                    Select::make('city_id')
                                        ->relationship('city', 'city_name')
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->visible(fn(Get $get) => filled($get('state_id')))
                                        ->options(
                                            fn(Get $get) =>
                                            \App\Models\Location\City::query()
                                                ->where('state_id', $get('state_id'))
                                                ->pluck('city_name', 'id')
                                        ),
                                ])
                                ->collapsible(),
                        ]),
                ])
        ];
    }
}
