<?php

namespace App\Filament\Resources\Job\JobRoleResource\Pages;

use Filament\Resources\Pages\Page;
use Filament\Resources\Pages\Concerns\InteractsWithRecord;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\IconColumn;
use Filament\Forms\Components\Select;
use App\Models\Job\JobRole;
use Illuminate\Support\Str;
use Filament\Forms\Form;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use App\Filament\Resources\Job\JobRoleResource;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Forms\Get;
use Filament\Forms\Set;


class JobRoles extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = JobRoleResource::class;

    protected static string $view = 'filament.resources.job.job-role-resource.pages.job-roles';

    public ?array $data = [];
    public ?JobRole $jobRole = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('job_role_name')
                    ->label('Job Role')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->jobRole && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                Select::make('parent_id')
                    ->label('Parent')
                    ->options(JobRole::all()->pluck('job_role_name', 'id'))
                    ->searchable()
                    ->preload(),
                RichEditor::make('job_role_description')
                    ->disableToolbarButtons([
                        'codeBlock',
                    ])
                    ->label('Description')
                    ->columnSpanFull(),
                Toggle::make('is_active')
                    ->label('Active')
                    ->default(true),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(JobRole::query())
            ->columns([
                TextColumn::make('job_role_name')
                    ->label('Name')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                TextColumn::make('job_roles_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (JobRole $record): string => route('job.role', ['slug' => $record->slug]))
                        ->hidden(fn(JobRole $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(JobRole $record): bool => $record->trashed())
                        ->action(function (JobRole $record): void {
                            $this->jobRole = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['job_roles_count'])
                        ->hidden(fn(JobRole $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(JobRole $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),

                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                return $query->select('job_roles.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('jobs')
                                ->whereColumn('jobs.job_role_id', 'job_roles.id')
                                ->selectRaw('COUNT(*)');
                        },
                        'job_roles_count'
                    );
            });
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->jobRole) {
            // Update existing tag
            $this->jobRole->update($data);
            $message = 'Job Role updated successfully';
        } else {
            // Create new tag
            JobRole::create($data);
            $message = 'Job Role created successfully';
        }

        $this->jobRole = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->jobRole = null;
        $this->data = [];
        $this->form->fill();
    }
}
