<?php

namespace App\Filament\Resources\Job\JobSkillResource\Pages;

use Filament\Resources\Pages\Page;
use Filament\Resources\Pages\Concerns\InteractsWithRecord;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\IconColumn;
use App\Models\Job\JobSkill;
use Illuminate\Support\Str; 
use Filament\Forms\Form;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use App\Filament\Resources\Job\JobSkillResource;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Forms\Get;
use Filament\Forms\Set;


class JobSkills extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = JobSkillResource::class;

    protected static string $view = 'filament.resources.job.job-skill-resource.pages.job-skills';

    public ?array $data = [];
    public ?JobSkill $jobSkill = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('job_skill_name')
                    ->label('Job Skill')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->jobSkill && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                RichEditor::make('job_skill_description')
                    ->label('Description')
                    ->disableToolbarButtons([
                        'codeBlock',
                    ])
                    ->columnSpanFull(),
                Toggle::make('is_active')
                    ->label('Active')
                    ->default(true),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(JobSkill::query())
            ->columns([
                TextColumn::make('job_skill_name')
                    ->label('Name')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                TextColumn::make('job_skills_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->color('info')
                        ->icon('heroicon-o-eye')
                        //->url(fn (JobSkill $record): string => route('job.skill', ['slug' => $record->slug]))
                        ->hidden(fn(JobSkill $record): bool => $record->trashed())
                        ->openUrlInNewTab(),
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(JobSkill $record): bool => $record->trashed())
                        ->action(function (JobSkill $record): void {
                            $this->jobSkill = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['job_skills_count'])
                        ->hidden(fn(JobSkill $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(JobSkill $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                return $query->select('job_skills.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('job_job_skills')
                                ->whereColumn('job_job_skills.job_skill_id', 'job_skills.id')
                                ->selectRaw('COUNT(*)');
                        },
                        'job_skills_count'
                    );
            });
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->jobSkill) {
            // Update existing tag
            $this->jobSkill->update($data);
            $message = 'Job Skill updated successfully';
        } else {
            // Create new tag
            JobSkill::create($data);
            $message = 'Job Skill created successfully';
        }

        $this->jobSkill = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->jobSkill = null;
        $this->data = [];
        $this->form->fill();
    }
}
