<?php

namespace App\Filament\Resources\Lead;

use App\Enums\LeadFormFieldDataType;
use App\Enums\LeadFormFieldType;
use App\Filament\Resources\Lead\LeadFormFieldsResource\Pages;
use App\Models\Lead\LeadFormFields;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Toggle;

class LeadFormFieldsResource extends Resource
{
    protected static ?string $model = LeadFormFields::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // Using 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                Section::make('Basic Information')
                                    ->schema([
                                        TextInput::make('field_name')
                                            ->required()
                                            ->maxLength(100)
                                            ->placeholder('Enter field name')
                                            ->helperText('The label that will be displayed for this field'),
                                        TextInput::make('field_id')
                                            ->required()
                                            ->maxLength(100)
                                            ->placeholder('Enter field ID')
                                            ->helperText('Unique identifier for this field'),
                                        Select::make('field_type')
                                            ->enum(LeadFormFieldType::class)
                                            ->options(LeadFormFieldType::class)
                                            ->required()
                                            ->default(LeadFormFieldType::Text)
                                            ->native(false)
                                            ->helperText('The type of input field to display'),

                                        Select::make('data_type')
                                            ->enum(LeadFormFieldDataType::class)
                                            ->options(LeadFormFieldDataType::class)
                                            ->required()
                                            ->default(LeadFormFieldDataType::AlphaNumeric)
                                            ->native(false)
                                            ->helperText('The type of data this field will accept'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('field_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('field_id')
                    ->searchable(),
                Tables\Columns\TextColumn::make('field_type')
                    ->searchable(),
                Tables\Columns\TextColumn::make('data_type')
                    ->searchable(),
                Tables\Columns\TextColumn::make('forms_count')
                    ->counts('forms')
                    ->label('Forms Using')
                    ->badge()
                    ->color(fn ($state) => $state > 0 ? 'success' : 'gray')
                    ->tooltip(fn ($record) => $record->forms_count > 0
                        ? 'Used in ' . $record->forms_count . ' form(s)'
                        : 'Not used in any forms'),
                Tables\Columns\IconColumn::make('is_default')
                    ->boolean()
                    ->tooltip('Default fields cannot be edited or deleted'),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\EditAction::make()
                        ->hidden(fn ($record) => $record->is_default),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn (LeadFormFields $record) =>
                            $record->is_default || $record->forms()->count() > 0
                        )
                        ->tooltip(fn (LeadFormFields $record) =>
                            $record->forms()->count() > 0
                                ? 'Cannot delete: Field is used in ' . $record->forms()->count() . ' form(s)'
                                : null
                        ),
                ]),
            ])
            ->bulkActions([
                // Removing bulk actions completely as they were only for deletion
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\LeadFormFields::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
