<?php

namespace App\Filament\Resources\Lead\LeadFormFieldsResource\Pages;

use App\Enums\LeadFormFieldDataType;
use App\Enums\LeadFormFieldType;
use App\Filament\Resources\Lead\LeadFormFieldsResource;
use App\Models\Lead\LeadFormFields as LeadFormFieldsModel;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\DB;

class LeadFormFields extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = LeadFormFieldsResource::class;

    protected static string $view = 'filament.resources.lead.lead-form-fields-resource.pages.lead-form-fields';

    public ?array $data = [];

    public ?LeadFormFieldsModel $leadFormField = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->schema([
                        Section::make('Basic Information')
                            ->schema([
                                TextInput::make('field_name')
                                    ->required()
                                    ->maxLength(100)
                                    ->placeholder('Enter field name')
                                    ->helperText('The label that will be displayed for this field'),
                                TextInput::make('field_id')
                                    ->required()
                                    ->maxLength(100)
                                    ->placeholder('Enter field ID')
                                    ->helperText('Unique identifier for this field')
                                    ->unique(ignoreRecord: true),
                                Select::make('field_type')
                                    ->enum(LeadFormFieldType::class)
                                    ->options(LeadFormFieldType::class)
                                    ->required()
                                    ->default(LeadFormFieldType::Text)
                                    ->native(false)
                                    ->helperText('The type of input field to display'),
                                Select::make('data_type')
                                    ->enum(LeadFormFieldDataType::class)
                                    ->options(LeadFormFieldDataType::class)
                                    ->required()
                                    ->default(LeadFormFieldDataType::AlphaNumeric)
                                    ->native(false)
                                    ->helperText('The type of data this field will accept'),     
                            ]),
                    ]),
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(LeadFormFieldsModel::query())
            ->columns([
                TextColumn::make('field_name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('field_id')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('field_type')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('data_type')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('forms_count')
                    ->counts('forms')
                    ->label('Forms Using')
                    ->badge()
                    ->color(fn ($state) => $state > 0 ? 'success' : 'gray')
                    ->tooltip(fn ($record) => $record->forms_count > 0
                        ? 'Used in ' . $record->forms_count . ' form(s)'
                        : 'Not used in any forms'),
                IconColumn::make('is_default')
                    ->boolean()
                    ->tooltip('Default fields cannot be edited or deleted'),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn (LeadFormFieldsModel $record): bool => $record->trashed())
                        ->action(function (LeadFormFieldsModel $record): void {
                            $this->leadFormField = $record;
                            $this->data = $record->toArray();
                        }),
                    DeleteAction::make()
                        ->hidden(fn (LeadFormFieldsModel $record): bool =>
                            $record->trashed() ||
                            $record->is_default ||
                            $record->forms()->count() > 0
                        ),
                    ForceDeleteAction::make()
                        ->hidden(fn (LeadFormFieldsModel $record): bool =>
                            $record->is_default ||
                            $record->forms()->count() > 0
                        )
                        ->before(function (ForceDeleteAction $action, LeadFormFieldsModel $record) {
                            if ($record->forms()->count() > 0) {
                                Notification::make()
                                    ->danger()
                                    ->title('Cannot Delete Field')
                                    ->body('This field is currently being used in ' . $record->forms()->count() . ' form(s). Please remove it from all forms before deleting.')
                                    ->persistent()
                                    ->send();

                                $action->halt();
                            }
                        }),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->before(function (DeleteBulkAction $action, $records) {
                            foreach ($records as $record) {
                                if ($record->forms()->count() > 0) {
                                    Notification::make()
                                        ->danger()
                                        ->title('Cannot Delete Fields')
                                        ->body('Some selected fields are being used in forms and cannot be deleted.')
                                        ->persistent()
                                        ->send();

                                    $action->halt();
                                    return;
                                }
                            }
                        }),
                    ForceDeleteBulkAction::make()
                        ->before(function (ForceDeleteBulkAction $action, $records) {
                            foreach ($records as $record) {
                                if ($record->forms()->count() > 0) {
                                    Notification::make()
                                        ->danger()
                                        ->title('Cannot Delete Fields')
                                        ->body('Some selected fields are being used in forms and cannot be deleted.')
                                        ->persistent()
                                        ->send();

                                    $action->halt();
                                    return;
                                }
                            }
                        }),
                    RestoreBulkAction::make(),
                ]),
            ]);
    }

    public function save(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $data = $this->form->getState();

            if ($this->leadFormField) {
                $this->leadFormField->update($data);
                $message = 'Form field updated successfully';
            } else {
                LeadFormFieldsModel::create($data);
                $message = 'Form field created successfully';
            }

            DB::commit();

            $this->leadFormField = null;
            $this->data = [];
            $this->form->fill();

            Notification::make()
                ->success()
                ->title($message)
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->danger()
                ->title('Error saving form field')
                ->body('Please try again.')
                ->send();
        }
    }

    public function cancelEdit(): void
    {
        $this->leadFormField = null;
        $this->data = [];
        $this->form->fill();
    }
}
