<?php

namespace App\Filament\Resources\Lead;

use App\Enums\LeadFormLayout;
use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use App\Filament\Resources\Lead\LeadFormResource\Pages;
use App\Filament\Resources\Lead\LeadFormResource\RelationManagers\FormFieldsRelationManager;
use App\Models\Lead\LeadForm;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;

class LeadFormResource extends Resource
{
    protected static ?string $model = LeadForm::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // Using 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7])
                            ->schema([
                                Section::make('Basic Information')
                                    ->schema([
                                        TextInput::make('title')
                                            ->required()
                                            ->maxLength(255)
                                            ->placeholder('Enter form title')
                                            ->helperText('The title of your lead form'),

                                        Textarea::make('description')
                                            ->maxLength(500)
                                            ->placeholder('Enter form description')
                                            ->helperText('A brief description of the form purpose'),

                                        Textarea::make('submit_message')
                                            ->maxLength(500)
                                            ->placeholder('Enter submission success message')
                                            ->helperText('Message to show after successful form submission'),
                                    ]),

                                Section::make('Form Steps')
                                    ->description('Define the steps for your wizard form')
                                    ->schema([
                                        Repeater::make('form_steps')
                                            ->label('Steps')
                                            ->schema([
                                                Hidden::make('uuid')
                                                    ->default(fn () => (string) Str::ulid()),

                                                TextInput::make('name')
                                                    ->label('Step Name')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->placeholder('Enter step name (e.g., contact-info)')
                                                    ->helperText('Internal identifier for the step')
                                                    ->columnSpan(12),

                                                TextInput::make('title')
                                                    ->label('Step Title')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->placeholder('Enter step title')
                                                    ->columnSpan(12),

                                                Textarea::make('subtitle')
                                                    ->label('Step Subtitle')
                                                    ->maxLength(500)
                                                    ->placeholder('Enter step subtitle (optional)')
                                                    ->columnSpan(12),

                                                Toggle::make('is_active')
                                                    ->label('Active')
                                                    ->default(true)
                                                    ->helperText('Enable or disable this step')
                                                    ->inline(false)
                                                    ->columnSpan(12),

                                                FileUpload::make('image')
                                                    ->label('Step Icon')
                                                    ->image()
                                                    ->directory('lead_form')
                                                    ->maxSize(5120)
                                                    ->imageEditor()
                                                    ->imageResizeMode('cover')
                                                    ->imagePreviewHeight('80')
                                                    ->helperText('Upload an icon or image for this step')
                                                    ->columnSpan(12),
                                            ])
                                            ->columns(12)
                                            ->reorderable()
                                            ->collapsible()
                                            ->collapsed()
                                            ->itemLabel(fn (array $state): ?string => $state['title'] ?? $state['name'] ?? 'New Step')
                                            ->addActionLabel('Add Step')
                                            ->defaultItems(0),
                                    ])
                                    ->visible(fn (Get $get): bool => $get('form_layout') === 'wizard'),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3])
                            ->schema([
                                Section::make('Form Settings')
                                    ->schema([
                                        Select::make('match_criteria')
                                            ->enum(LeadFormMatchCriteria::class)
                                            ->options(LeadFormMatchCriteria::class)
                                            ->required()
                                            ->default(LeadFormMatchCriteria::Category)
                                            ->native(false)
                                            ->helperText('How to match leads with categories'),

                                        Select::make('reception_mode')
                                            ->enum(LeadFormReceptionMode::class)
                                            ->options(LeadFormReceptionMode::class)
                                            ->required()
                                            ->default(LeadFormReceptionMode::Manual)
                                            ->native(false)
                                            ->helperText('How leads should be processed'),

                                        Toggle::make('enable_captcha')
                                            ->required()
                                            ->default(true)
                                            ->helperText('Enable CAPTCHA protection'),

                                        Toggle::make('is_active')
                                            ->required()
                                            ->default(true)
                                            ->helperText('Form is available for submissions'),
                                        Select::make('form_layout')
                                            ->enum(LeadFormLayout::class)
                                            ->options(LeadFormLayout::class)
                                            ->required()
                                            ->default(LeadFormLayout::Simple)
                                            ->searchable()
                                            ->preload(),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->sortable()
                    ->formatStateUsing(function ($state) {
                        return Str::limit($state, 20);
                    }),
                Tables\Columns\TextColumn::make('match_criteria')
                    ->badge()
                    ->color(fn (LeadFormMatchCriteria $state): string => match ($state) {
                        LeadFormMatchCriteria::Category => 'info',
                        LeadFormMatchCriteria::CategoryLocation => 'warning',
                    }),
                Tables\Columns\TextColumn::make('reception_mode')
                    ->badge()
                    ->color(fn (LeadFormReceptionMode $state): string => match ($state) {
                        LeadFormReceptionMode::Auto => 'success',
                        LeadFormReceptionMode::Manual => 'info',
                    }),
                Tables\Columns\TextColumn::make('visits_count')
                    ->label('Visits')
                    ->counts('visits')
                    ->sortable()
                    ->badge()
                    ->color('info'),
                Tables\Columns\TextColumn::make('leads_count')
                    ->label('Leads')
                    ->counts('leads')
                    ->sortable()
                    ->badge()
                    ->color('success'),
                Tables\Columns\IconColumn::make('enable_captcha')
                    ->boolean(),
                Tables\Columns\IconColumn::make('is_active')
                    ->boolean(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalDescription('Are you sure you want to delete this lead form? This action cannot be undone.'),
                ])
                    ->tooltip('Actions')
                    ->icon('heroicon-m-ellipsis-vertical'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            FormFieldsRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLeadForms::route('/'),
            'view' => Pages\ViewLeadFormCustom::route('/{record}'),
            'edit' => Pages\EditLeadForm::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
