<?php

namespace App\Filament\Resources\Lead\LeadFormResource\Pages;

use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use App\Enums\LeadFormLayout;
use App\Filament\Resources\Lead\LeadFormResource;
use App\Models\Lead\LeadForm;
use App\Models\Lead\LeadFormFields;
use Filament\Actions;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Resources\Pages\ListRecords;

class ListLeadForms extends ListRecords
{
    protected static string $resource = LeadFormResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\CreateAction::make('new_lead_form')
                ->label('New Lead Form')
                ->modalHeading('Create New Lead Form')
                ->modalDescription('Create a new lead form with basic settings.')
                ->form([
                    Section::make('Basic Information')
                        ->columns(12)
                        ->schema([
                            TextInput::make('title')
                                ->required()
                                ->maxLength(255)
                                ->placeholder('Enter form title')
                                ->columnSpanFull(),

                            Textarea::make('description')
                                ->maxLength(500)
                                ->placeholder('Enter form description')
                                ->columnSpanFull(),

                            Textarea::make('submit_message')
                                ->maxLength(500)
                                ->placeholder('Enter submission success message')
                                ->columnSpanFull(),

                            Select::make('match_criteria')
                                ->enum(LeadFormMatchCriteria::class)
                                ->options(LeadFormMatchCriteria::class)
                                ->required()
                                ->default(LeadFormMatchCriteria::Category)
                                ->native(false)
                                ->columnSpan(4),

                            Select::make('reception_mode')
                                ->enum(LeadFormReceptionMode::class)
                                ->options(LeadFormReceptionMode::class)
                                ->required()
                                ->default(LeadFormReceptionMode::Manual)
                                ->native(false)
                                ->columnSpan(4),
                            Select::make('form_layout')
                                ->enum(LeadFormLayout::class)
                                ->options(LeadFormLayout::class)
                                ->default(LeadFormLayout::Simple)
                                ->searchable()
                                ->preload()
                                ->columnSpan(4),
                            Toggle::make('enable_captcha')
                                ->required()
                                ->default(true)
                                ->inline(false)
                                ->columnSpan(6),

                            Toggle::make('is_active')
                                ->required()
                                ->default(true)
                                ->inline(false)
                                ->columnSpan(6),

                        ]),
                ])
                ->after(function (LeadForm $record) {
                    // Attach default fields to the newly created LeadForm
                    $defaultFields = LeadFormFields::where('is_default', true)
                        ->orderBy('id') // Order by ID to ensure consistent ordering
                        ->get();

                    $order = 1; // Start ordering from 1

                    foreach ($defaultFields as $field) {
                        $record->fieldLeadForm()->create([
                            'field_id' => $field->id,
                            'field_placeholder' => $field->field_name, // Use field name as default placeholder
                            'field_columns' => 12, // Default to full width
                            'is_required' => true, // Default to not required
                            'is_visible' => true, // Default to visible
                            'field_order' => $order++, // Sequential ordering starting from 1
                            'field_options' => null,
                            'toggle_yes_text' => 'Yes',
                            'toggle_no_text' => 'No',
                            'form_layout' => $record->form_layout,
                            'field_data' => [
                                'container_class' => null,
                                'container_id' => null,
                                'data_columns' => null,
                            ],
                        ]);
                    }
                })
                ->successRedirectUrl(fn(LeadForm $record): string => LeadFormResource::getUrl('edit', ['record' => $record])),
        ];
    }
}
