<?php

namespace App\Filament\Resources\Lead\LeadFormResource\Pages;

use App\Filament\Resources\Lead\LeadFormResource;
use Filament\Resources\Pages\ViewRecord;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Actions\Action;
use App\Models\Lead\Lead;

class ViewLeadFormCustom extends ViewRecord implements HasTable
{
    use InteractsWithTable;

    protected static string $resource = LeadFormResource::class;

    protected static string $view = 'filament.resources.lead.lead-form-resource.pages.view-lead-form-custom';

    protected function mutateFormDataBeforeFill(array $data): array
    {
        // Load the relationships with necessary data
        $this->record->load([
            'formFields' => function ($query) {
                $query->orderBy('field_lead_form.field_order', 'asc');
            },
            'leads' => function ($query) {
                $query->latest()->limit(10);
            },
            'memberCategories',
            'author',
            'visits'
        ]);

        // Load counts separately for better performance
        $this->record->loadCount(['visits', 'leads']);

        return $data;
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('viewAllLeads')
                ->label('View All Leads')
                ->icon('heroicon-o-eye')
                ->color('primary')
                ->url(fn () => \App\Filament\Resources\Lead\LeadResource::getUrl('index', [
                    'tableFilters' => [
                        'leadForm' => [
                            'value' => $this->record->id
                        ]
                    ]
                ])),
            Action::make('edit')
                ->label('Edit Form')
                ->icon('heroicon-o-pencil')
                ->url(fn () => LeadFormResource::getUrl('edit', ['record' => $this->record])),
        ];
    }

    public function getLatestLeads(): Builder
    {
        return Lead::query()
            ->where('lead_form_id', $this->record->id)
            ->latest()
            ->limit(10);
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getLatestLeads())
            ->columns([
                TextColumn::make('first_name')
                    ->label('Name')
                    ->formatStateUsing(fn ($record) => trim($record->first_name . ' ' . $record->last_name))
                    ->searchable(['first_name', 'last_name'])
                    ->sortable(),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->copyMessage('Email copied')
                    ->icon('heroicon-m-envelope'),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->sortable()
                    ->getStateUsing(fn ($record) => $record->phone ?? 'N/A'),

                TextColumn::make('created_at')
                    ->label('Submitted')
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->description(fn ($record) => $record->created_at->diffForHumans()),
            ])
            ->actions([
                \Filament\Tables\Actions\ViewAction::make()
                    ->url(fn ($record) => \App\Filament\Resources\Lead\LeadResource::getUrl('view', ['record' => $record]))
                    ->openUrlInNewTab(false),
            ])
            ->emptyStateHeading('No Leads Yet')
            ->emptyStateDescription('No leads have been submitted through this form yet.')
            ->emptyStateIcon('heroicon-o-document-text')
            ->striped()
            ->defaultPaginationPageOption(10);
    }
}
