<?php

namespace App\Filament\Resources\Lead\LeadFormResource\RelationManagers;

use App\Enums\FieldDataSource;
use App\Enums\FileUploadType;
use App\Enums\LeadFormFieldDataType;
use App\Enums\LeadFormFieldType;
use App\Models\Lead\LeadFormFields;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class FormFieldsRelationManager extends RelationManager
{
    protected static string $relationship = 'fieldLeadForm';

    protected static ?string $recordTitleAttribute = 'field.field_name';

    /**
     * Get field type for the selected field_id
     */
    private function getFieldType(?string $fieldId): ?LeadFormFieldType
    {
        if (! $fieldId) {
            return null;
        }

        $field = LeadFormFields::find($fieldId);

        return $field?->field_type;
    }

    /**
     * Get all enum classes from app/Enums directory
     */
    private function getEnumOptions(): array
    {
        $enumPath = app_path('Enums');
        $enums = [];

        if (! is_dir($enumPath)) {
            return $enums;
        }

        $files = glob($enumPath.'/*.php');

        foreach ($files as $file) {
            $className = basename($file, '.php');
            $fullClassName = "App\\Enums\\{$className}";

            if (class_exists($fullClassName) && enum_exists($fullClassName)) {
                $enums[$fullClassName] = \Illuminate\Support\Str::title(
                    \Illuminate\Support\Str::of($className)->snake(' ')
                );
            }
        }

        return $enums;
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 1,
                    ])
                    ->schema([
                        // Left Column - Field Configuration
                        Group::make()
                            ->schema([
                                Section::make('Field Configuration')
                                    ->schema([
                                        Select::make('field_id')
                                            ->label('Field')
                                            ->options(LeadFormFields::all()->pluck('field_name', 'id'))
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->live()
                                            ->reactive(),

                                        TextInput::make('field_label')
                                            ->label('Field Label')
                                            ->maxLength(191)
                                            ->helperText('Label text displayed above the field'),

                                        Select::make('form_step')
                                            ->label('Form Step')
                                            ->options(function (): array {
                                                $leadForm = $this->ownerRecord;
                                                if ($leadForm && $leadForm->form_layout === \App\Enums\LeadFormLayout::Wizard && $leadForm->form_steps) {
                                                    return collect($leadForm->form_steps)
                                                        ->pluck('title', 'uuid')
                                                        ->toArray();
                                                }

                                                return [];
                                            })
                                            ->searchable()
                                            ->preload()
                                            ->visible(function (): bool {
                                                $leadForm = $this->ownerRecord;

                                                return $leadForm &&
                                                       $leadForm->form_layout === \App\Enums\LeadFormLayout::Wizard &&
                                                       ! empty($leadForm->form_steps);
                                            })
                                            ->helperText('Select which step this field belongs to'),

                                        TextInput::make('field_placeholder')
                                            ->label('Placeholder Text')
                                            ->maxLength(191)
                                            ->helperText('Text to show as placeholder in the field'),

                                        Textarea::make('help_text')
                                            ->label('Help Text')
                                            ->maxLength(500)
                                            ->helperText('Additional help text for the field'),

                                        Grid::make()
                                            ->columns(2)
                                            ->schema([
                                                Select::make('field_columns')
                                                    ->label('Field Width')
                                                    ->options([
                                                        1 => '1/12 (Tiny)',
                                                        2 => '2/12 (Small)',
                                                        3 => '3/12 (Quarter)',
                                                        4 => '4/12 (Third)',
                                                        6 => '6/12 (Half)',
                                                        8 => '8/12 (Two Thirds)',
                                                        12 => '12/12 (Full)',
                                                    ])
                                                    ->default(12)
                                                    ->required(),

                                                TextInput::make('field_order')
                                                    ->label('Display Order')
                                                    ->numeric()
                                                    ->default(0)
                                                    ->helperText('Order of field display (0 = first)'),
                                            ]),
                                    ]),
                            ]),

                        // Right Column - Additional Settings
                        Group::make()
                            ->schema([
                                Section::make('Field Settings')
                                    ->schema([
                                        Grid::make()
                                            ->columns(2)
                                            ->schema([
                                                Toggle::make('is_required')
                                                    ->label('Required Field')
                                                    ->default(false)
                                                    ->helperText('Field must be filled out'),

                                                Toggle::make('is_visible')
                                                    ->label('Visible')
                                                    ->default(true)
                                                    ->helperText('Field is visible on form'),
                                            ]),

                                        TextInput::make('toggle_yes_text')
                                            ->label('Toggle Yes Text')
                                            ->maxLength(50)
                                            ->helperText('Text for "Yes" option (toggle fields only)')
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::Toggle),

                                        TextInput::make('toggle_no_text')
                                            ->label('Toggle No Text')
                                            ->maxLength(50)
                                            ->helperText('Text for "No" option (toggle fields only)')
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::Toggle),

                                        // Data Source selector
                                        Select::make('data_source')
                                            ->label('Data Source')
                                            ->options(FieldDataSource::class)
                                            ->helperText('Choose how to provide options for this field')
                                            ->live()
                                            ->reactive()
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::Select ||
                                                $this->getFieldType($get('field_id')) === LeadFormFieldType::Radio ||
                                                $this->getFieldType($get('field_id')) === LeadFormFieldType::Checkbox
                                            ),

                                        // Enum class selector - visible when data_source is 'enum'
                                        Select::make('field_options')
                                            ->label('Enum Class')
                                            ->options($this->getEnumOptions())
                                            ->searchable()
                                            ->preload()
                                            ->helperText('Select an enum class to use as options')
                                            ->afterStateHydrated(function (Select $component, $state) {
                                                // Only set state if it's a string (enum class path)
                                                if (is_array($state)) {
                                                    $component->state(null);
                                                }
                                            })
                                            ->dehydrateStateUsing(fn ($state) => is_string($state) ? $state : null)
                                            ->visible(fn (Get $get): bool => $get('data_source') === FieldDataSource::Enum->value &&
                                                (
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Select ||
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Radio ||
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Checkbox
                                                )
                                            ),

                                        // Key-Value options for Select fields - visible when data_source is 'list' or not set
                                        KeyValue::make('field_options')
                                            ->label('Select Options')
                                            ->helperText('Define key-value pairs for select options (key = value, value = display text)')
                                            ->keyLabel('Option Value')
                                            ->valueLabel('Display Text')
                                            ->addActionLabel('Add Option')
                                            ->reorderable()
                                            ->visible(fn (Get $get): bool => ($get('data_source') === FieldDataSource::List->value || $get('data_source') === null) &&
                                                (
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Select ||
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Radio ||
                                                    $this->getFieldType($get('field_id')) === LeadFormFieldType::Checkbox
                                                )
                                            ),

                                        // Simple textarea for Radio fields
                                        Textarea::make('field_options')
                                            ->label('Textarea')
                                            ->helperText('Enter options for radio fields (one per line or JSON array)')
                                            ->rows(3)
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::Textarea),

                                        // Media upload for section logo (Spatie Media Library)
                                        SpatieMediaLibraryFileUpload::make('section_logo')
                                            ->collection('section_logo')
                                            ->label('Section Logo')
                                            ->helperText('Optional logo shown with this field section')
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::SectionTitle),

                                        // Container settings stored under field_data JSON
                                        Section::make('Container Settings')
                                            ->schema([
                                                Group::make()
                                                    ->columns(2)
                                                    ->schema([
                                                        TextInput::make('field_data.container_class')
                                                            ->label('Container Class')
                                                            ->maxLength(191)
                                                            ->helperText('Custom CSS classes for the field container'),

                                                        TextInput::make('field_data.container_id')
                                                            ->label('Container ID')
                                                            ->maxLength(191)
                                                            ->helperText('Custom ID attribute for the field container'),

                                                        Select::make('field_data.data_columns')
                                                            ->label('Data Columns')
                                                            ->options([
                                                                1 => 'One Column',
                                                                2 => 'Two Columns',
                                                                3 => 'Three Columns',
                                                                4 => 'Four Columns',
                                                            ])
                                                            ->visible(function (Get $get): bool {
                                                                return $this->getFieldType($get('field_id')) === LeadFormFieldType::Checkbox;
                                                            })
                                                            ->helperText('Number of option columns when using checkbox field type'),
                                                    ]),
                                            ])
                                            ->columns(1),

                                        // File Upload Validation Settings
                                        Section::make('File Upload Settings')
                                            ->schema([
                                                Select::make('field_data.file_validation.allowed_types')
                                                    ->label('Allowed File Types')
                                                    ->options(collect(FileUploadType::cases())->mapWithKeys(fn ($case) => [
                                                        $case->value => $case->label(),
                                                    ]))
                                                    ->multiple()
                                                    ->searchable()
                                                    ->preload()
                                                    ->default(['image'])
                                                    ->required()
                                                    ->helperText('Select one or more file types to allow'),

                                                Grid::make()
                                                    ->columns(2)
                                                    ->schema([
                                                        TextInput::make('field_data.file_validation.max_size_kb')
                                                            ->label('Max File Size (KB)')
                                                            ->numeric()
                                                            ->default(2048)
                                                            ->required()
                                                            ->minValue(1)
                                                            ->maxValue(102400)
                                                            ->helperText('Maximum file size in kilobytes (e.g., 2048 = 2MB)'),

                                                        TextInput::make('field_data.file_validation.max_files')
                                                            ->label('Maximum Number of Files')
                                                            ->numeric()
                                                            ->default(5)
                                                            ->required()
                                                            ->minValue(1)
                                                            ->maxValue(20)
                                                            ->helperText('Maximum number of files that can be uploaded'),
                                                    ]),
                                            ])
                                            ->visible(fn (Get $get): bool => $this->getFieldType($get('field_id')) === LeadFormFieldType::File)
                                            ->collapsed()
                                            ->columns(1),

                                        // Conditional Logic (Dependent Fields)
                                        Section::make('Conditional Logic')
                                            ->schema([
                                                Toggle::make('conditional_logic.enabled')
                                                    ->label('Is Dependent Field?')
                                                    ->helperText('This field will only show when a specific value is selected in another field')
                                                    ->live()
                                                    ->reactive(),

                                                Select::make('conditional_logic.parent_field_id')
                                                    ->label('Parent Field')
                                                    ->options(function (Get $get): array {
                                                        $leadFormId = $this->ownerRecord->id;
                                                        $currentFormStep = $get('form_step');
                                                        $currentFieldId = $get('id');

                                                        return \App\Models\Lead\FieldLeadForm::query()
                                                            ->where('lead_form_id', $leadFormId)
                                                            ->when($currentFormStep, fn ($q) => $q->where('form_step', $currentFormStep))
                                                            ->when($currentFieldId, fn ($q) => $q->where('id', '!=', $currentFieldId))
                                                            ->whereHas('field', function ($query) {
                                                                $query->whereIn('field_type', [
                                                                    LeadFormFieldType::Select,
                                                                    LeadFormFieldType::Radio,
                                                                    LeadFormFieldType::Checkbox,
                                                                    LeadFormFieldType::Toggle,
                                                                ]);
                                                            })
                                                            ->with('field')
                                                            ->get()
                                                            ->mapWithKeys(function ($fieldLeadForm) {
                                                                $label = $fieldLeadForm->field_label ?? $fieldLeadForm->field->field_name;

                                                                return [$fieldLeadForm->id => $label];
                                                            })
                                                            ->toArray();
                                                    })
                                                    ->searchable()
                                                    ->preload()
                                                    ->live()
                                                    ->reactive()
                                                    ->visible(fn (Get $get): bool => (bool) $get('conditional_logic.enabled'))
                                                    ->helperText('Select the field this depends on')
                                                    ->afterStateUpdated(fn (callable $set) => $set('conditional_logic.parent_option_value', null))
                                                    ->rules([
                                                        function (Get $get) {
                                                            return function (string $attribute, $value, \Closure $fail) use ($get) {
                                                                if (! $get('conditional_logic.enabled') || ! $value) {
                                                                    return;
                                                                }

                                                                // Check nesting level
                                                                $parent = \App\Models\Lead\FieldLeadForm::find($value);
                                                                if ($parent && $parent->isConditional()) {
                                                                    $level = $parent->getDependencyLevel();
                                                                    if ($level >= 2) {
                                                                        $fail('Maximum nesting level (3 levels) reached. Cannot add more nested dependencies.');
                                                                    }
                                                                }
                                                            };
                                                        },
                                                    ]),

                                                Select::make('conditional_logic.parent_option_value')
                                                    ->label('Show When Value Is')
                                                    ->options(function (Get $get): array {
                                                        $parentFieldId = $get('conditional_logic.parent_field_id');

                                                        if (! $parentFieldId) {
                                                            return [];
                                                        }

                                                        $parentField = \App\Models\Lead\FieldLeadForm::find($parentFieldId);

                                                        if (! $parentField) {
                                                            return [];
                                                        }

                                                        // Handle Toggle fields
                                                        if ($parentField->field->field_type === LeadFormFieldType::Toggle) {
                                                            return [
                                                                '1' => $parentField->toggle_yes_text ?? 'Yes',
                                                                '0' => $parentField->toggle_no_text ?? 'No',
                                                            ];
                                                        }

                                                        // Handle Enum data source
                                                        if ($parentField->data_source === 'enum' && is_string($parentField->field_options)) {
                                                            $enumClass = $parentField->field_options;
                                                            if (class_exists($enumClass) && enum_exists($enumClass)) {
                                                                return collect($enumClass::cases())
                                                                    ->mapWithKeys(fn ($case) => [
                                                                        $case->value => method_exists($case, 'label') ? $case->label() : $case->name,
                                                                    ])
                                                                    ->toArray();
                                                            }
                                                        }

                                                        // Handle regular field options (array)
                                                        if (is_array($parentField->field_options)) {
                                                            return $parentField->field_options;
                                                        }

                                                        return [];
                                                    })
                                                    ->searchable()
                                                    ->preload()
                                                    ->visible(fn (Get $get): bool => (bool) $get('conditional_logic.enabled') && (bool) $get('conditional_logic.parent_field_id'))
                                                    ->helperText('This field will show when the parent field has this value selected')
                                                    ->required(fn (Get $get): bool => (bool) $get('conditional_logic.enabled')),
                                            ])
                                            ->collapsed()
                                            ->columns(1),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('field.field_name')
            ->columns([
                Tables\Columns\TextColumn::make('field.field_name')
                    ->label('Field Name')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('field.field_id')
                    ->label('Field ID')
                    ->searchable()
                    ->sortable(),

                // TextColumn::make('field.field_type')
                //     ->label('Type')
                //     ->badge()
                //     ->color(fn (LeadFormFieldType $state): string => $state->getColor()),

                // TextColumn::make('field.data_type')
                //     ->label('Data Type')
                //     ->badge()
                //     ->color(fn (LeadFormFieldDataType $state): string => $state->getColor()),

                TextColumn::make('form_step')
                    ->label('Step')
                    ->formatStateUsing(function ($state, $record): ?string {
                        if (! $state || ! $record->leadForm) {
                            return null;
                        }
                        $step = collect($record->leadForm->form_steps ?? [])
                            ->firstWhere('uuid', $state);

                        return $step['title'] ?? $state;
                    })
                    ->badge()
                    ->color('info')
                    ->visible(function (): bool {
                        return $this->ownerRecord &&
                               $this->ownerRecord->form_layout === \App\Enums\LeadFormLayout::Wizard &&
                               ! empty($this->ownerRecord->form_steps);
                    }),

                TextColumn::make('field_order')
                    ->label('Order')
                    ->sortable(),

                TextColumn::make('field_columns')
                    ->label('Width')
                    ->formatStateUsing(fn ($state) => $state.'/12'),

                IconColumn::make('is_required')
                    ->label('Required')
                    ->boolean(),

                // IconColumn::make('is_visible')
                //     ->label('Visible')
                //     ->boolean(),

                // IconColumn::make('field.is_default')
                //     ->label('Default')
                //     ->boolean()
                //     ->tooltip('Default fields cannot be deleted'),
            ])
            ->filters([
                TernaryFilter::make('is_required')
                    ->label('Required Fields')
                    ->trueLabel('Required only')
                    ->falseLabel('Optional only')
                    ->native(false),

                TernaryFilter::make('is_visible')
                    ->label('Visible Fields')
                    ->trueLabel('Visible only')
                    ->falseLabel('Hidden only')
                    ->native(false),

                SelectFilter::make('field_type')
                    ->preload()
                    ->label('Field Type')
                    ->options(LeadFormFieldType::class)
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['value'],
                            fn (Builder $query, $value): Builder => $query->whereHas('field', function (Builder $query) use ($value) {
                                $query->where('field_type', $value);
                            })
                        );
                    })
                    ->native(false),
            ])
            ->headerActions([
                CreateAction::make()
                    ->label('Add Field to Form')
                    ->form(fn (Form $form): Form => $this->form($form))
                    ->mutateFormDataUsing(function (array $data): array {
                        $data['lead_form_id'] = $this->ownerRecord->id;

                        return $data;
                    }),
            ])
            ->actions([
                ActionGroup::make([
                    EditAction::make()
                        ->form(fn (Form $form): Form => $this->form($form)),
                    DeleteAction::make()
                        ->hidden(fn ($record) => $record->field?->is_default)
                        ->requiresConfirmation()
                        ->modalDescription('Are you sure you want to remove this field from the form?'),
                ]),

            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalDescription('Are you sure you want to remove the selected fields from the form?'),
                ]),
            ])
            ->emptyStateActions([
                CreateAction::make()
                    ->label('Add First Field')
                    ->form(fn (Form $form): Form => $this->form($form)),
            ])
            ->defaultSort('field_order', 'asc')
            ->defaultPaginationPageOption(25)
            ->reorderable('field_order');
    }
}
