<?php

namespace App\Filament\Resources\Lead;

use App\Filament\Resources\Lead\LeadResource\Pages;
use App\Filament\Resources\Lead\LeadResource\RelationManagers;
use App\Models\Lead\Lead;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Section;
use App\Enums\LeadStatus;

class LeadResource extends Resource
{
    protected static ?string $model = Lead::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Lead Information')
                    ->schema([
                        Forms\Components\Select::make('lead_form_id')
                            ->relationship('leadForm', 'title')
                            ->searchable()
                            ->preload()
                            ->columnSpanFull(),
                        Forms\Components\TextInput::make('lead_source')
                            ->prefix('https://')
                            ->columnSpanFull(),
                    ])
                    ->columns(2),
                Section::make('Form Fields')
                    ->schema([
                        Forms\Components\Repeater::make('lead_data')
                            ->schema([
                                Forms\Components\TextInput::make('key'),
                                Forms\Components\TextInput::make('value'),
                            ])
                            ->columns(2),
                    ])
                    ->columns(1),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('leadForm.title')
                    ->numeric()
                    ->getStateUsing(fn($record) => $record->leadForm->title ?? 'N/A')
                    ->sortable(),

                Tables\Columns\TextColumn::make('first_name')
                    ->label('Name')
                    ->formatStateUsing(fn($record) => trim($record->first_name . ' ' . $record->last_name))
                    ->searchable(['first_name', 'last_name'])
                    ->sortable(),
                Tables\Columns\TextColumn::make('email')
                    ->sortable(),
                Tables\Columns\TextColumn::make('phone')
                    ->sortable()
                    ->searchable()
                    ->getStateUsing(fn($record) => $record->phone ?? 'N/A'),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime('M d, Y H:i A')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('leadForm')
                    ->relationship('leadForm', 'title')
                    ->searchable()
                    ->preload()
                    ->label('Lead Form'),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                // Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\UsersRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLeads::route('/'),
            'create' => Pages\CreateLead::route('/create'),
            'view' => Pages\ViewLeadCustom::route('/{record}'),
            // 'edit' => Pages\EditLead::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
