<?php

namespace App\Filament\Resources\Lead\LeadResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\BadgeColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Select;
use App\Models\User\User;
use App\Enums\LeadStatus;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Actions\ActionGroup;
use Filament\Support\Enums\FontWeight;
use Filament\Tables\Columns\ViewColumn;

class UsersRelationManager extends RelationManager
{
    protected static string $relationship = 'users';

    protected static ?string $recordTitleAttribute = 'name';

    protected static ?string $pluralRecordTitle = 'Assigned Users';

    protected static ?string $title = 'Assigned Users';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('User Assignment')
                    ->description('Assign a user to this lead and set their status')
                    ->icon('heroicon-o-user-plus')
                    ->schema([
                                                    Select::make('user_id')
                                ->relationship(
                                    name: 'users', 
                                    titleAttribute: 'name',
                                    modifyQueryUsing: fn (Builder $query) => $query->active()
                                )
                            ->label('Select User')
                            ->searchable(['name', 'email'])
                            ->preload()
                            ->required()
                            ->getOptionLabelFromRecordUsing(fn (User $record): string => "{$record->name} ({$record->email})")
                            ->columnSpanFull(),
                        
                        Select::make('lead_status')
                            ->label('Lead Status')
                            ->options(LeadStatus::class)
                            ->default(LeadStatus::Pending->value)
                            ->required()
                            ->columnSpanFull()
                            ->helperText('Set the initial status for this user\'s assignment to the lead'),
                        
                        Textarea::make('notes')
                            ->label('Assignment Notes')
                            ->placeholder('Add any notes about this assignment...')
                            ->rows(3)
                            ->columnSpanFull(),
                    ])
                    ->columns(2),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->heading('Assigned Users')
            ->description('Manage users assigned to this lead and track their individual status')
            ->columns([
                ImageColumn::make('profile_photo_path')
                    ->label('Avatar')
                    ->circular()
                    ->defaultImageUrl(fn ($record) => 'https://ui-avatars.com/api/?name=' . urlencode($record->name) . '&color=7F9CF5&background=EBF4FF')
                    ->size(40),
                
                TextColumn::make('name')
                    ->label('User Name')
                    ->searchable()
                    ->sortable()
                    ->weight(FontWeight::Medium)
                    ->description(fn ($record) => $record->email),
                
                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->copyMessage('Email copied!')
                    ->icon('heroicon-m-envelope')
                    ->toggleable(),
                
                BadgeColumn::make('pivot.lead_status')
                    ->label('Status')
                    ->formatStateUsing(fn (string $state): string => LeadStatus::from($state)->getLabel())
                    ->colors([
                        'warning' => LeadStatus::Pending->value,
                        'primary' => LeadStatus::FollowUp->value,
                        'success' => LeadStatus::SoldOut->value,
                        'danger' => LeadStatus::Closed->value,
                        'gray' => [LeadStatus::BadLead->value, LeadStatus::Trashed->value],
                        'info' => LeadStatus::Duplicate->value,
                    ])
                    ->icons([
                        'heroicon-o-clock' => LeadStatus::Pending->value,
                        'heroicon-o-arrow-path' => LeadStatus::FollowUp->value,
                        'heroicon-o-banknotes' => LeadStatus::SoldOut->value,
                        'heroicon-o-archive-box' => LeadStatus::Closed->value,
                        'heroicon-o-x-circle' => LeadStatus::BadLead->value,
                        'heroicon-o-document-duplicate' => LeadStatus::Duplicate->value,
                        'heroicon-o-trash' => LeadStatus::Trashed->value,
                    ])
                    ->sortable(),
                
                TextColumn::make('phone')
                    ->label('Phone')
                    ->icon('heroicon-m-phone')
                    ->copyable()
                    ->copyMessage('Phone copied!')
                    ->toggleable(isToggledHiddenByDefault: true),
                
                TextColumn::make('company_name')
                    ->label('Company')
                    ->icon('heroicon-m-building-office')
                    ->toggleable(isToggledHiddenByDefault: true),
                
                TextColumn::make('created_at')
                    ->label('User Since')
                    ->dateTime('M j, Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('pivot.lead_status')
                    ->label('Lead Status')
                    ->options(LeadStatus::class)
                    ->multiple()
                    ->preload(),
            ])
            ->headerActions([
                Tables\Actions\AttachAction::make()
                    ->label('Assign User')
                    ->color('primary')
                    ->icon('heroicon-o-user-plus')
                    ->modalHeading('Assign User to Lead')
                    ->modalSubmitActionLabel('Assign User')
                    ->form(fn (Tables\Actions\AttachAction $action): array => [
                        $action->getRecordSelect()
                            ->searchable(['name', 'email'])
                            ->preload()
                            ->getOptionLabelFromRecordUsing(fn (User $record): string => "{$record->name} ({$record->email})")
                            ->optionsLimit(50),
                        
                        Select::make('lead_status')
                            ->label('Initial Status')
                            ->options(LeadStatus::class)
                            ->default(LeadStatus::Pending->value)
                            ->required()
                            ->helperText('Set the initial status for this user\'s assignment'),
                    ])
                    ->preloadRecordSelect()
                    ->successNotificationTitle('User assigned successfully'),
            ])
            ->actions([
                ActionGroup::make([
                    Tables\Actions\EditAction::make()
                        ->label('Update Status')
                        ->icon('heroicon-o-pencil-square')
                        ->modalHeading('Update Lead Status')
                        ->modalSubmitActionLabel('Update Status')
                        ->form([
                            Select::make('lead_status')
                                ->label('Lead Status')
                                ->options(LeadStatus::class)
                                ->required()
                                ->helperText('Update the status for this user\'s assignment'),
                        ])
                        ->successNotificationTitle('Status updated successfully'),
                    
                    Tables\Actions\ViewAction::make()
                        ->label('View User')
                        ->icon('heroicon-o-eye')
                        ->url(fn ($record) => route('filament.admin.resources.users.view', $record))
                        ->openUrlInNewTab(),
                    
                    Tables\Actions\DetachAction::make()
                        ->label('Remove Assignment')
                        ->icon('heroicon-o-user-minus')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('Remove User Assignment')
                        ->modalDescription('Are you sure you want to remove this user from the lead? This action cannot be undone.')
                        ->modalSubmitActionLabel('Remove Assignment')
                        ->successNotificationTitle('User removed from lead'),
                ])
                ->label('Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->size('sm')
                ->color('gray')
                ->button(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make()
                        ->label('Remove Selected')
                        ->requiresConfirmation()
                        ->modalHeading('Remove Selected Users')
                        ->modalDescription('Are you sure you want to remove the selected users from this lead?')
                        ->modalSubmitActionLabel('Remove Users')
                        ->successNotificationTitle('Selected users removed from lead'),
                    
                    Tables\Actions\BulkAction::make('updateStatus')
                        ->label('Update Status')
                        ->icon('heroicon-o-arrow-path')
                        ->color('primary')
                        ->form([
                            Select::make('lead_status')
                                ->label('New Status')
                                ->options(LeadStatus::class)
                                ->required(),
                        ])
                        ->action(function (array $data, $records) {
                            foreach ($records as $record) {
                                $record->pivot->update(['lead_status' => $data['lead_status']]);
                            }
                        })
                        ->successNotificationTitle('Status updated for selected users'),
                ]),
            ])
            ->defaultSort('name', 'asc')
            ->striped()
            ->paginated([10, 25, 50])
            ->poll('30s')
            ->emptyStateHeading('No users assigned')
            ->emptyStateDescription('Start by assigning users to this lead to track their individual progress.')
            ->emptyStateIcon('heroicon-o-users');
    }
}
