<?php

namespace App\Filament\Resources\Location;

use App\Models\Location\Area;
use App\Models\Location\City;
use App\Models\Location\State;
use App\Models\Location\Country;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Str;
use App\Filament\Resources\Location\AreaResource\Pages;

class AreaResource extends Resource
{
    protected static ?string $model = Area::class;

    protected static ?string $navigationIcon = 'heroicon-o-map-pin';

    protected static ?string $navigationGroup = 'Location Management';

    protected static ?string $recordTitleAttribute = 'name';

    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg'      => 10, // Adjusting to a 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7]) // Change to 7 for 70%
                            ->schema([
                                Section::make('Area Details')
                                    ->schema([
                                        TextInput::make('name')
                                            ->label('Area Name')
                                            ->placeholder('Enter the service area name')
                                            ->required()
                                            ->autofocus()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                                if ($context === 'create' && empty($get('slug'))) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('slug')
                                            ->label('Slug')
                                            ->placeholder('Enter the area slug')
                                            ->helperText('Will be automatically generated from area name if left empty.')
                                            ->maxLength(255)
                                            ->required()
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (?string $state, Set $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            })
                                            ->dehydrated(),
                                    ]),
                                // SEO Section
                                // static::getSeoSection(),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3]) // Change to 3 for 30%
                            ->schema([
                                Section::make('Location')
                                    ->schema([
                                        Select::make('city_id')
                                            ->label('City')
                                            ->helperText('Select the city this service area belongs to')
                                            ->relationship('city', 'city_name')
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->live(),
                                    ]),
                            ]),

                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Area Name')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('city.city_name')
                    ->label('City')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('userLocations_count')
                    ->label('Used by Locations')
                    ->counts('userLocations')
                    ->badge()
                    ->color('info'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label('Updated')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('city_id')
                    ->label('City')
                    ->relationship('city', 'city_name')
                    ->searchable()
                    ->preload()
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('name');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListAreas::route('/'),
            'create' => Pages\CreateArea::route('/create'),
            'view'   => Pages\ViewArea::route('/{record}'),
            'edit'   => Pages\EditArea::route('/{record}/edit'),
        ];
    }
}
