<?php

namespace App\Filament\Resources\Location;

use App\Filament\Resources\Location\CityResource\Pages;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use Filament\Forms\Components\Builder;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use App\Models\User\MemberCategory;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Actions\ActionGroup;

class CityResource extends Resource
{
    protected static ?string $model = City::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-office-2';

    protected static ?string $navigationGroup = 'Location Management';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make(['default' => 1, 'lg' => 12]) // Define grid structure
                    ->schema([
                        Group::make()
                            ->schema([
                                Section::make('Page Details')
                                    ->schema([
                                        TextInput::make('heading')
                                            ->label('Heading')
                                            ->required(),
                                        RichEditor::make('description')
                                            ->label('Description'),
                                    ]),
                                Section::make('Sections')
                                    ->schema([
                                        Builder::make('page_data.components')
                                            ->label('Components')
                                            ->blocks([
                                                Builder\Block::make('content')
                                                    ->label('City Content')
                                                    ->schema([
                                                        Hidden::make('component')
                                                            ->default('frontend.components.city.content'),
                                                        TextInput::make('heading')->required(),
                                                        Textarea::make('description'),
                                                        Repeater::make('list')
                                                            ->schema([
                                                                TextInput::make('item')->required(),
                                                            ])
                                                            ->grid(1),
                                                    ]),
                                                Builder\Block::make('tags')
                                                    ->label('City Tags')
                                                    ->schema([
                                                        Hidden::make('component')
                                                            ->default('frontend.components.city.tags'),
                                                        TextInput::make('heading')->required(),
                                                        Textarea::make('description'),
                                                        Repeater::make('links')
                                                            ->schema([
                                                                TextInput::make('link_text')->required(),
                                                                TextInput::make('link_url')->required()->url(),
                                                            ])
                                                            ->grid(1),
                                                    ]),

                                                Builder\Block::make('categories')
                                                    ->label('City Categories')
                                                    ->schema([
                                                        Hidden::make('component')
                                                            ->default('frontend.components.city.categories'),
                                                        TextInput::make('heading')->required(),
                                                        Textarea::make('description'),
                                                        Select::make('categories')
                                                            ->options(MemberCategory::all()->pluck('category_name', 'slug'))
                                                            ->multiple()
                                                            ->searchable(),
                                                    ]),
                                                Builder\Block::make('list')
                                                    ->label('City List Detail')
                                                    ->schema([
                                                        Hidden::make('component')
                                                            ->default('frontend.components.city.list-detail'),
                                                        TextInput::make('heading')->required(),
                                                        Textarea::make('description'),
                                                        Repeater::make('items')
                                                            ->itemLabel(
                                                                fn(array $state): ?string =>
                                                                isset($state['heading']) ? $state['heading'] : null
                                                            )
                                                            ->schema([
                                                                TextInput::make('heading')->required(),
                                                                Select::make('icon')
                                                                    ->options([
                                                                        'drop-fill' => 'Drop Icon',
                                                                        'services-search' => 'Services Search',
                                                                        'land-icon' => 'Land Icon',
                                                                    ]),
                                                                Repeater::make('list')
                                                                    ->label('Items')
                                                                    ->schema([
                                                                        TextInput::make('item'),
                                                                    ])
                                                                    ->grid(1)
                                                                    ->itemLabel(fn(array $state): ?string => $state['item'])
                                                                    ->collapsible()
                                                                    ->collapsed(),
                                                            ])
                                                            ->maxItems(3)
                                                            ->collapsible()
                                                            ->collapsed()
                                                            ->grid(1),
                                                    ]),
                                            ])
                                            ->collapsible()
                                            ->collapsed()
                                            ->columnSpanFull(),
                                    ]),

                            ])
                            ->columnSpan(['lg' => 8]),
                        Group::make()
                            ->schema([
                                Section::make('City Details')
                                    ->schema([
                                        TextInput::make('city_name')
                                            ->label('City Name')
                                            ->placeholder('Enter City Name')
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('slug')
                                            ->label('Slug')
                                            ->placeholder('Slug')
                                            ->required()
                                            ->maxLength(255),

                                        Select::make('state_id')
                                            ->label('State')
                                            ->options(State::all()->pluck('state_name', 'id'))
                                            ->reactive(),
                                        Select::make('country_id')
                                            ->label('Country')
                                            ->options(Country::all()->pluck('country_name', 'id'))
                                            ->reactive(),

                                    ]),
                                Section::make("Featured Image")
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('featured_image')
                                            ->collection('featured_image')
                                            ->image()
                                            ->imageEditor()

                                    ]),

                            ])
                            ->columnSpan(['lg' => 4]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('city_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('state.state_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('country.country_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->searchable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                ActionGroup::make([
                    Tables\Actions\Action::make('View')
                        ->url(fn(City $record): string => route('city.show', $record->slug))
                        ->icon('heroicon-o-eye'),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make(),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListCities::route('/'),
            'create' => Pages\CreateCity::route('/create'),
            'edit'   => Pages\EditCity::route('/{record}/edit'),
        ];
    }
}
