<?php

namespace App\Filament\Resources\Manufacturer;

use App\Filament\Resources\Manufacturer\ManufacturerResource\Pages;
use App\Filament\Resources\Manufacturer\ManufacturerResource\RelationManagers;
use App\Models\User\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Checkbox;
use App\Enums\UserProfileStatus;
use App\Enums\UserProfileType;
use Filament\Forms\Get;

class ManufacturerResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 12, // Using 12-column grid
                    ])
                    ->schema([
                        Group::make()
                            ->columnSpan(['lg' => 8])
                            ->schema([
                                Section::make('Personal Information')
                                    ->schema([
                                        TextInput::make('name')
                                            ->label('Full Name')
                                            ->hidden()
                                            ->dehydrated()
                                            ->live()
                                            ->afterStateUpdated(function (string $operation, $state, Get $get) {
                                                return $get('profile.first_name') . ' ' . $get('profile.last_name');
                                            }),
                                        TextInput::make('profile.first_name')
                                            ->label('First Name')
                                            ->required()
                                            ->live()
                                            ->maxLength(255),
                                        TextInput::make('profile.last_name')
                                            ->label('Last Name')
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('email')
                                            ->label('Email')
                                            ->email()
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.phone')
                                            ->label('Phone')
                                            ->tel()
                                            ->mask('(999) 999-9999')
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.fax')
                                            ->label('Fax')
                                            ->mask('(999) 999-9999')
                                            ->tel()
                                            ->maxLength(255),
                                    ])
                                    ->columns(['lg' => 2]),
                                Section::make('Company Information')
                                    ->schema([
                                        TextInput::make('profile.company_name')
                                            ->label('Company Name')
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.website')
                                            ->label('Company Website')
                                            ->url()
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.year_established')
                                            ->label('Year Established')
                                            ->required()
                                            ->numeric(),
                                        Textarea::make('profile.about')
                                            ->label('About Company')
                                            ->required()
                                            ->maxLength(255),
                                        Textarea::make('profile.business_hours')
                                            ->label('Business Hours')
                                            ->required(),
                                        Textarea::make('profile.slogan')
                                            ->label('Slogan')
                                            ->required(),
                                        Textarea::make('profile.services')
                                            ->label('Services')
                                            ->required(),
                                        Textarea::make('profile.other_services')
                                            ->label('Other Services')
                                            ->required(),
                                    ])
                                    ->visible(fn(Get $get) => $get('profile.profile_type') === UserProfileType::Business->value),
                                Section::make('Social Media')
                                    ->schema([
                                        TextInput::make('profile.social_media_links.facebook')
                                            ->inlineLabel(true)
                                            ->label('Facebook')
                                            ->url()
                                            ->prefixIcon('fab-facebook'),
                                        TextInput::make('profile.social_media_links.instagram')
                                            ->inlineLabel(true)
                                            ->label('Instagram')
                                            ->url()
                                            ->prefixIcon('fab-instagram'),
                                        TextInput::make('profile.social_media_links.twitter')
                                            ->inlineLabel(true)
                                            ->label('Twitter')
                                            ->url()
                                            ->prefixIcon('fab-twitter'),
                                        TextInput::make('profile.social_media_links.linkedin')
                                            ->inlineLabel(true)
                                            ->label('LinkedIn')
                                            ->url()
                                            ->prefixIcon('fab-linkedin'),
                                        TextInput::make('profile.social_media_links.youtube')
                                            ->inlineLabel(true)
                                            ->label('Youtube')
                                            ->url()
                                            ->prefixIcon('fab-youtube'),
                                        TextInput::make('profile.social_media_links.tiktok')
                                            ->inlineLabel(true)
                                            ->label('TikTok')
                                            ->url()
                                            ->prefixIcon('fab-tiktok'),
                                        TextInput::make('profile.social_media_links.pinterest')
                                            ->inlineLabel(true)
                                            ->label('Pinterest')
                                            ->url()
                                            ->prefixIcon('fab-pinterest'),
                                    ]),
                                Section::make('Password')
                                    ->columns(['lg' => 2])
                                    ->schema([
                                        TextInput::make('password')
                                            ->label('Password')
                                            ->password()
                                            ->live()
                                            ->required(fn(string $operation): bool => $operation === 'create')
                                            ->dehydrated(fn($state) => filled($state))
                                            ->maxLength(255),
                                        TextInput::make('password_confirmation')
                                            ->label('Confirm Password')
                                            ->password()
                                            ->required(fn(Get $get): bool => filled($get('password')))
                                            ->maxLength(255),
                                        Checkbox::make('password_change_email')
                                            ->label('Send new password email to user.')
                                            ->default(false)
                                            ->hidden(fn(Get $get): bool => !filled($get('password'))),
                                    ]),
                            ]),
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                Section::make('Avatar')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('avatar')
                                            ->label('Upload Avatar'),

                                        Select::make('profile.status')
                                            ->options(UserProfileStatus::class)
                                            ->required()
                                            ->default(UserProfileStatus::Active->value)
                                            ->native(false),
                                        // Select::make('memberCategories') // The name of the relationship in the User model
                                        // ->relationship('memberCategories', 'category_name',modifyQueryUsing:function(Builder $query){
                                        //     return $query->limit(5);
                                        // })
                                        // ->multiple()
                                        // ->label('Member Categories') 
                                        // ->required(),
                                    ])
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime('d-m-Y'),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListManufacturers::route('/'),
            'create' => Pages\CreateManufacturer::route('/create'),
            'view' => Pages\ViewManufacturer::route('/{record}'),
            'edit' => Pages\EditManufacturer::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
        ->whereHas('roles', function (Builder $query) {
            $query->where('name', 'Manufacturer');
        })
        ->withoutGlobalScopes([
            SoftDeletingScope::class,
        ]);
    }
}
