<?php

namespace App\Filament\Resources\Member;

use App\Filament\Resources\Member\MemberCategoryResource\Pages;
use App\Filament\Resources\Member\MemberCategoryResource\RelationManagers;
use App\Models\User\MemberCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Tabs;
use App\Filament\Traits\HasSeoSection;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Illuminate\Support\Str; 
use Filament\Forms\Set;
use Filament\Forms\Get;

class MemberCategoryResource extends Resource
{
    use HasSeoSection;
    protected static ?string $model = MemberCategory::class;

    // protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Member';
    protected static ?string $navigationLabel = 'Member Category';
    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('Category')
                    ->tabs([
                        Tabs\Tab::make('Category')
                            ->schema([
                                TextInput::make('category_name')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                        if ($context === 'create' && empty($get('slug'))) {
                                            $set('slug', Str::slug($state));
                                        }
                                    }),
                                TextInput::make('slug')
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (string $state, Set $set) {
                                        $set('slug', Str::slug($state));
                                    })
                                    ->dehydrated()
                                    ->unique(ignoreRecord: true),
                                SpatieMediaLibraryFileUpload::make('category_logo')
                                    ->label('Category Logo')
                                    ->collection('category_logo')
                                    ->columnSpanFull(),

                                Select::make('parent_id')
                                    ->label('Parent Category')
                                    ->relationship(
                                        'parent',
                                        'category_name',
                                        fn (Builder $query) => $query->where('is_active', true)
                                    )
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Select parent category')
                                    ->nullable(),
                                RichEditor::make('category_description')
                                    ->columnSpanFull(),
                                Toggle::make('is_active')
                                    ->label('Visible')
                                    ->default(true),
                            ]),
                        Tabs\Tab::make('SEO')
                            ->schema([
                                static::getSeoSection(),
                            ]),
                    ])
                    ->columnSpanFull()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
        ->columns([
            TextColumn::make('category_name')
                ->searchable()
                ->sortable(),
            TextColumn::make('parent.category_name')
                ->label('Parent')
                ->searchable()
                ->sortable(false),
            TextColumn::make('slug')
                ->searchable()
                ->sortable(),
            TextColumn::make('users_count')
                ->label('Count')
                ->sortable(),
            ToggleColumn::make('is_active')
                ->label('Visible'),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('member_categories.*')
                ->selectSub(
                    function ($subQuery) {
                        $subQuery->from('member_category_user')
                                 ->selectRaw('COUNT(*)')
                                 ->whereColumn('member_category_user.member_category_id', 'member_categories.id');
                    },
                    'users_count' // Name of the additional column for the count
                );
                return $query;
            })
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('')
                    ->tooltip('Edit')
                    ->slideOver()
                    ->visible(fn ($record) => ! $record->trashed()),
                Tables\Actions\DeleteAction::make()
                    ->label('')
                    ->tooltip('Delete')
                    ->visible(fn ($record) => ! $record->trashed()),
                Tables\Actions\ForceDeleteAction::make()
                    ->label('')
                    ->tooltip('Delete Permanently')
                    ->visible(fn ($record) => $record->trashed()),
                Tables\Actions\RestoreAction::make()
                    ->label('')
                    ->tooltip('Restore')
                    ->visible(fn ($record) => $record->trashed()),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageMemberCategories::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
