<?php

namespace App\Filament\Resources\Member\MemberResource\Pages;

use App\Filament\Resources\Member\MemberResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use App\Enums\UserProfileStatus;
use App\Enums\UserProfileType;
use App\Models\User\MyRole;
use Spatie\Permission\Models\Role;
use Filament\Notifications\Notification;
use Illuminate\View\View;
use App\Services\User\BusinessHourService;

class CreateMember extends CreateRecord
{
    protected static string $resource = MemberResource::class;
    protected static ?string $title = 'Create Member';
    protected static ?string $pluralTitle = 'Create Member';

    protected function getFormActions(): array
    {
        return [
            $this->getCreateFormAction()
                ->label('Create')
                ->icon('heroicon-o-plus'),
            $this->getCreateAnotherFormAction()
                ->label('Create & New')
                ->icon('heroicon-o-plus-circle'),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label('Create')
                ->icon('heroicon-o-plus')
                ->action('create')
                ->color('primary')
                ->successNotification(
                    Notification::make()
                        ->success()
                        ->title('Member Created')
                        ->body('Your member has been created.')
                )
                ->keyBindings(['mod+s']),
            Actions\Action::make('createAnother')
                ->label('Create & New')
                ->icon('heroicon-o-plus-circle')
                ->action('createAnother')
                ->color('gray')
                ->successNotification(
                    Notification::make()
                        ->success()
                        ->title('Member Created')
                        ->body('Your member has been created. Create another one!')
                ),
            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn() => MemberResource::getUrl()),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        if (!empty($data['profile']['first_name']) && !empty($data['profile']['last_name'])) {
            $data['name'] = $data['profile']['first_name'] . ' ' . $data['profile']['last_name'];
        }
        return $data;
    }

    protected function afterCreate(): void
    {
        $profile = $this->data['profile'];
        $profile['profile_type'] = UserProfileType::Business->value;
        $this->record->profile()->create($profile);
        $this->record->refresh();

        // Persist profile_tags to meta table if provided
        if (isset($this->data['profile']['meta']['profile_tags'])) {
            $this->record->profile->setMeta('profile_tags', $this->data['profile']['meta']['profile_tags']);
            $this->record->profile->save();
        }
        if (isset($this->data['profile']['meta']['faqs'])) {
            $this->record->profile->setMeta('faqs', $this->data['profile']['meta']['faqs']);
            $this->record->profile->save();
        }


        $role = MyRole::where('name', 'Service')->first();
        $this->record->assignRole($role->name);

        // Create default business hours if not provided
        $businessHourService = app(BusinessHourService::class);
        if ($this->record->businessHours()->count() === 0) {
            $businessHourService->createDefaultBusinessHours($this->record);
        }
    }
}
