<?php

namespace App\Filament\Resources\Member\MemberResource\Pages;

use App\Filament\Resources\Member\MemberResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use Illuminate\View\View;
use App\Enums\UserProfileStatus;
use App\Services\User\BusinessHourService;

class EditMember extends EditRecord
{
    protected static string $resource = MemberResource::class;
    protected static ?string $title = 'Edit Member';
    protected static ?string $pluralTitle = 'Edit Member';

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title('Member Updated')
            ->body('Your member has been updated successfully.');
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),
            Actions\Action::make('services')
                ->label('Manage Services')
                ->icon('heroicon-o-briefcase')
                ->color('info')
                ->url(fn() => route('filament.espadmin.resources.member.members.services', ['record' => $this->record]))
                ->visible(fn(): bool => !$this->record->trashed()),
            Actions\ViewAction::make()
                ->label('View')
                ->icon('heroicon-o-eye')
                ->url(fn() => route('filament.espadmin.resources.member.members.view', ['record' => $this->record]))
                ->color('success'),
            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    Actions\RestoreAction::make()
                        ->label('Restore')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->visible(fn(): bool => $this->record->trashed())
                        ->modalHeading('Restore Member')
                        ->modalDescription('Are you sure you want to restore this member? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, restore it')
                        ->before(function () {
                            $this->record->profile()->update(['status' => UserProfileStatus::Active]);
                        }),

                    Actions\ForceDeleteAction::make()
                        ->label('Force Delete')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->visible(fn(): bool => $this->record->trashed())
                        ->modalHeading('Force Delete Member')
                        ->modalDescription('Are you sure you want to force delete this member? This action cannot be undone.')
                        ->modalSubmitActionLabel('Yes, force delete it')
                        ->before(function () {
                            $this->record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                        }),

                    Actions\DeleteAction::make()
                        ->label('Move to Trash')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->hidden(fn(): bool => $this->record->trashed())
                        ->modalHeading('Delete Member')
                        ->modalDescription('Are you sure you want to delete this member? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, delete it')
                        ->before(function () {
                            $this->record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                        })
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Member deleted')
                                ->body('The member has been moved to trash.')
                        ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn() => MemberResource::getUrl()),
            ])
                ->tooltip('More Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('info')
                ->button()
                ->extraAttributes([
                    'class' => 'more-actions-btn',
                ]),

        ];
    }

    public function getFormActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),
            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn() => MemberResource::getUrl()),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        // Use array representation to avoid indirect modification warnings
        $data['profile'] = $this->record->profile?->toArray() ?? [];
        // Load profile_tags from meta so the TagsInput shows existing values
        $data['profile']['meta']['profile_tags'] = $this->record->profile?->getMeta('profile_tags') ?? [];
        $data['profile']['meta']['faqs'] = $this->record->profile?->getMeta('faqs') ?? [];
        
        
        // Ensure all 7 days of business hours exist
        $businessHourService = app(BusinessHourService::class);
        if ($this->record->businessHours()->count() < 7) {
            $existingDays = $this->record->businessHours->pluck('day_of_week')->map(fn($day) => $day->value)->toArray();
            $allDays = collect(\App\Enums\DayOfWeek::cases())->map(fn($day) => $day->value)->toArray();
            $missingDays = array_diff($allDays, $existingDays);
            
            foreach ($missingDays as $day) {
                $this->record->businessHours()->create([
                    'day_of_week' => $day,
                    'is_open' => true,
                    'hours_label' => 'Regular Hours',
                    'open_time' => '09:00',
                    'close_time' => '17:00',
                ]);
            }
            $this->record->refresh();
        }
        
        return $data;
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        if (!empty($data['profile']['first_name']) && !empty($data['profile']['last_name'])) {
            $data['name'] = $data['profile']['first_name'] . ' ' . $data['profile']['last_name'];
        }
        return $data;
    }

    protected function afterSave(): void
    {
        $this->record->profile()->updateOrCreate(
            ['user_id' => $this->record->id],
            $this->data['profile']
        );
        // Persist profile_tags to meta table
        if (isset($this->data['profile']['meta']['profile_tags'])) {
            $this->record->profile->setMeta('profile_tags', $this->data['profile']['meta']['profile_tags']);
            $this->record->profile->save();
        }

        if (isset($this->data['profile']['meta']['faqs'])) {
            $this->record->profile->setMeta('faqs', $this->data['profile']['meta']['faqs']);
            $this->record->profile->save();
        }
    }
}
