<?php

namespace App\Filament\Resources\Member\MemberResource\Pages;

use App\Filament\Resources\Member\MemberResource;
use App\Models\User\User;
use App\Services\User\UserServiceService;
use Filament\Resources\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Actions\Action;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\Rule;

class MemberServices extends Page implements HasForms, HasActions
{
    use InteractsWithForms;
    use InteractsWithActions;

    protected static string $resource = MemberResource::class;

    protected static string $view = 'filament.resources.member.pages.member-services';

    public User $record;

    #[Rule('required|min:3|max:191')]
    public string $service_title = '';

    #[Rule('required|min:10')]
    public string $service_description = '';

    public $services = [];
    public $editingService = null;

    public $profileMetaForm = [];

    public function mount($record): void
    {
        $this->record = $record;
        $userServiceService = app(UserServiceService::class);
        $this->loadServices($userServiceService);
        $this->loadProfileMeta();
    }

    public function loadProfileMeta()
    {
        $profile = $this->record->profile;

        if (!$profile) {
            Notification::make()
                ->title('Error')
                ->body('Member profile not found.')
                ->danger()
                ->send();
            return;
        }

        $supportServicesValue = $profile->getMeta('support_services');
        $whyChooseUsValue = $profile->getMeta('why_choose_us');
        $emergencyServiceEnabled = $profile->getMeta('emergency_service_enabled');
        $emergencyServiceTitle = $profile->getMeta('emergency_service_title');
        $emergencyServiceDescription = $profile->getMeta('emergency_service_description');
        $emergencyServiceNumber = $profile->getMeta('emergency_service_number');

        if (empty($whyChooseUsValue)) {
            $whyChooseUsValue = [
                ['key' => '', 'value' => '']
            ];
        }

        $this->profileMetaForm = [
            'support_services' => $supportServicesValue ?: [],
            'why_choose_us' => $whyChooseUsValue ?: [],
            'emergency_service_enabled' => (bool)$emergencyServiceEnabled,
            'emergency_service_title' => $emergencyServiceTitle ?: '',
            'emergency_service_description' => $emergencyServiceDescription ?: '',
            'emergency_service_number' => $emergencyServiceNumber ?: '',
        ];

        $this->form->fill($this->profileMetaForm);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TagsInput::make('support_services')
                    ->rules(['max:5'])
                    ->label('Supported Services')
                    ->placeholder('Add a service and press Enter')
                    ->helperText('List the services this member specializes in or supports')
                    ->columnSpanFull(),

                Repeater::make('why_choose_us')
                    ->label('Why Choose This Member Points')
                    ->schema([
                        TextInput::make('key')
                            ->label('Title')
                            ->placeholder('e.g., "Expert Technicians"')
                            ->maxLength(30)
                            ->required(),
                        TextInput::make('value')
                            ->label('Description')
                            ->placeholder('e.g., "Our team has over 15 years of industry experience"')
                            ->maxLength(200)
                            ->rules(['required', 'string', 'max:200'])
                            ->required(),
                    ])
                    ->columns(2)
                    ->addActionLabel('Add Selling Point')
                    ->deleteAction(
                        fn(FormAction $action) => $action->requiresConfirmation()
                    )
                    ->columnSpanFull()
                    ->helperText('Add unique selling points to highlight why clients should choose this member\'s services')
                    ->collapsible()
                    ->defaultItems(false),

                Section::make('Emergency Services')
                    ->schema([
                        Checkbox::make('emergency_service_enabled')
                            ->label('Offer Emergency Services')
                            ->helperText('Enable if this member provides emergency services')
                            ->live(),

                        TextInput::make('emergency_service_title')
                            ->label('Emergency Service Title')
                            ->placeholder('e.g., "24/7 Emergency Support"')
                            ->maxLength(100)
                            ->hidden(fn(callable $get) => !$get('emergency_service_enabled'))
                            ->required(fn(callable $get) => $get('emergency_service_enabled')),

                        TextInput::make('emergency_service_number')
                            ->label('Emergency Contact Number')
                            ->mask('+1 (999) 999-9999')
                            ->maxLength(20)
                            ->hidden(fn(callable $get) => !$get('emergency_service_enabled'))
                            ->required(fn(callable $get) => $get('emergency_service_enabled')),
                    ])
                    ->collapsible(),
            ])
            ->statePath('profileMetaForm');
    }

    public function saveProfileMeta()
    {
        $data = $this->form->getState();
        $profile = $this->record->profile;

        if (!$profile) {
            Notification::make()
                ->title('Error')
                ->body('Member profile not found.')
                ->danger()
                ->send();
            return;
        }

        try {
            DB::beginTransaction();

            $profile->setMeta('support_services', $data['support_services']);

            if (!empty($data['why_choose_us'])) {
                $profile->setMeta('why_choose_us', $data['why_choose_us']);
            } else {
                $profile->setMeta('why_choose_us', []);
            }

            $profile->setMeta('emergency_service_enabled', $data['emergency_service_enabled']);

            if ($data['emergency_service_enabled']) {
                $profile->setMeta('emergency_service_title', $data['emergency_service_title'] ?? '');
                $profile->setMeta('emergency_service_description', $data['emergency_service_description'] ?? '');
                $profile->setMeta('emergency_service_number', $data['emergency_service_number'] ?? '');
            }

            $profile->save();

            DB::commit();

            $this->loadProfileMeta();

            Notification::make()
                ->title('Profile meta saved successfully')
                ->success()
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->title('Error saving profile meta')
                ->body('An error occurred while saving the member profile meta data: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    public function loadServices(UserServiceService $userServiceService)
    {
        $this->services = $userServiceService->getUserServices($this->record->id)
            ->orderBy('service_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function deleteAction(): Action
    {
        return Action::make('delete')
            ->requiresConfirmation()
            ->color('danger')
            ->button()
            ->label('Remove')
            ->modalIcon('heroicon-o-trash')
            ->modalIconColor('danger')
            ->modalHeading('Delete Service')
            ->modalDescription('Are you sure you want to delete this service? This action cannot be undone.')
            ->modalSubmitActionLabel('Yes, delete it')
            ->modalCancelActionLabel('No, cancel')
            ->mountUsing(fn(Action $action, array $arguments) => $action->arguments(['uuid' => $arguments['uuid']]))
            ->action(function (Action $action) {
                $arguments = $action->getArguments();
                if (isset($arguments['uuid'])) {
                    $this->deleteService($arguments['uuid']);
                }
            });
    }

    public function save(UserServiceService $userServiceService)
    {
        $this->validate();

        $data = [
            'user_id' => $this->record->id,
            'service_title' => $this->service_title,
            'service_description' => $this->service_description,
            'created_by' => auth()->id(),
        ];

        try {
            if ($this->editingService) {
                $userServiceService->updateByUuid($this->editingService['uuid'], $data);

                Notification::make()
                    ->title('Service updated successfully')
                    ->success()
                    ->send();
            } else {
                $userServiceService->create($data);

                Notification::make()
                    ->title('Service created successfully')
                    ->success()
                    ->send();
            }

            $this->resetForm();
            $this->loadServices($userServiceService);
            $this->dispatch('close-modal');
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('There was an error processing your request.')
                ->danger()
                ->send();
        }
    }

    public function editService(string $uuid, UserServiceService $userServiceService)
    {
        try {
            $this->editingService = $userServiceService->findByUuid($uuid);
            $this->service_title = $this->editingService->service_title;
            $this->service_description = $this->editingService->service_description;

            $this->dispatch('open-modal', 'add-service-modal');
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('Service not found.')
                ->danger()
                ->send();
        }
    }

    protected function deleteService(string $uuid)
    {
        try {
            $userServiceService = app(UserServiceService::class);
            $service = $userServiceService->findByUuid($uuid);

            if ($service) {
                $userServiceService->delete($service->id);

                Notification::make()
                    ->title('Service deleted successfully')
                    ->success()
                    ->send();

                $this->loadServices($userServiceService);
            }
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('There was an error deleting the service.')
                ->danger()
                ->send();
        }
    }

    public function resetForm(): void
    {
        $this->reset(['service_title', 'service_description', 'editingService']);
    }

    public function reorder($items): void
    {
        try {
            if (!is_array($items)) {
                throw new \Exception('Invalid data format');
            }

            DB::beginTransaction();

            $userServiceService = app(UserServiceService::class);
            $services = collect($items)->map(function ($item) use ($userServiceService) {
                return [
                    'uuid' => $item['value'],
                    'service' => $userServiceService->findByUuid($item['value'])
                ];
            })->filter(function ($item) {
                return !is_null($item['service']);
            });

            foreach ($services as $index => $item) {
                $item['service']->update([
                    'service_order' => $index + 1
                ]);
            }

            DB::commit();

            $this->loadServices($userServiceService);

            Notification::make()
                ->title('Order updated successfully')
                ->success()
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->title('Error')
                ->body('There was an error updating the order.')
                ->danger()
                ->send();
        }
    }

    public static function getNavigationLabel(): string
    {
        return 'Services';
    }

    public function getTitle(): string
    {
        return 'Member Services - ' . $this->record->name;
    }

    public function getBreadcrumb(): string
    {
        return 'Services';
    }
}
