<?php

namespace App\Filament\Resources\Member\MemberResource\Pages;

use App\Filament\Resources\Member\MemberResource;
use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;
use Livewire\Attributes\Computed;
use App\Enums\UserProfileStatus;
use Filament\Notifications\Notification;

class ViewMember extends ViewRecord
{
    protected static string $resource = MemberResource::class;

    protected static string $view = 'filament.resources.member.pages.custom-view-member';

    public string $activeTab = 'Overview';

    public function setActiveTab(string $tab): void
    {
        $this->activeTab = $tab;
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make()
                ->icon('heroicon-m-pencil-square'),
            Actions\Action::make('services')
                ->label('View Services')
                ->icon('heroicon-m-briefcase')
                ->color('info')
                ->url(fn() => route('filament.espadmin.resources.member.members.services', ['record' => $this->record]))
                ->visible(fn(): bool => !$this->record->trashed()),
            Actions\ActionGroup::make([
                Actions\Action::make('activate')
                    ->label(
                        fn() =>
                        $this->record->profile?->status === UserProfileStatus::PendingApproval
                            ? 'Approve'
                            : 'Activate'
                    )
                    ->icon('heroicon-m-check-circle')
                    ->color('success')
                    ->requiresConfirmation()
                    ->visible(
                        fn() => !$this->record->profile ||
                            in_array($this->record->profile->status, [
                                UserProfileStatus::NotActive,
                                UserProfileStatus::PendingApproval,
                                UserProfileStatus::OnHold,
                                UserProfileStatus::Incomplete,
                                UserProfileStatus::PastDue,
                                UserProfileStatus::Cancelled
                            ])
                    )
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::Active]);
                        Notification::make()
                            ->success()
                            ->title('Profile Activated')
                            ->body('The profile has been activated successfully.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('markNotActive')
                    ->label('Mark Not Active')
                    ->icon('heroicon-m-exclamation-circle')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->visible(fn() => !$this->record->profile || $this->record->profile->status !== UserProfileStatus::NotActive)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::NotActive]);
                        Notification::make()
                            ->warning()
                            ->title('Profile Not Active')
                            ->body('The profile has been marked as not active.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('markPendingApproval')
                    ->label('Mark Pending Approval')
                    ->icon('heroicon-m-clock')
                    ->color('info')
                    ->requiresConfirmation()
                    ->visible(fn() => !$this->record->profile || $this->record->profile->status !== UserProfileStatus::PendingApproval)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::PendingApproval]);
                        Notification::make()
                            ->info()
                            ->title('Profile Pending Approval')
                            ->body('The profile has been marked as pending approval.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('deactivate')
                    ->label('Put On Hold')
                    ->icon('heroicon-m-pause-circle')
                    ->color('gray')
                    ->requiresConfirmation()
                    ->visible(fn() => $this->record->profile && $this->record->profile->status === UserProfileStatus::Active)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::OnHold]);
                        Notification::make()
                            ->warning()
                            ->title('Profile On Hold')
                            ->body('The profile has been put on hold.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('markIncomplete')
                    ->label('Mark as Incomplete')
                    ->icon('heroicon-m-exclamation-triangle')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->visible(fn() => !$this->record->profile || $this->record->profile->status !== UserProfileStatus::Incomplete)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::Incomplete]);
                        Notification::make()
                            ->warning()
                            ->title('Profile Incomplete')
                            ->body('The profile has been marked as incomplete.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('markPastDue')
                    ->label('Mark Past Due')
                    ->icon('heroicon-m-currency-dollar')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->visible(fn() => !$this->record->profile || $this->record->profile->status !== UserProfileStatus::PastDue)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::PastDue]);
                        Notification::make()
                            ->danger()
                            ->title('Profile Past Due')
                            ->body('The profile has been marked as past due.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
                Actions\Action::make('cancel')
                    ->label('Cancel Account')
                    ->icon('heroicon-m-x-circle')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->visible(fn() => !$this->record->profile || $this->record->profile->status !== UserProfileStatus::Cancelled)
                    ->action(function () {
                        $this->record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                        Notification::make()
                            ->danger()
                            ->title('Account Cancelled')
                            ->body('The account has been cancelled successfully.')
                            ->send();
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $this->record]), navigate: true);
                    }),
            ])
                ->label('Change Status')
                ->icon('heroicon-m-user-circle')
                ->color('primary')
                ->button()
                ->hidden(fn(): bool => $this->record->trashed())
                ->dropdownPlacement('bottom-end')
                ->extraAttributes([
                    'class' => 'relative'
                ])
        ];
    }

    #[Computed]
    public function tabs(): array
    {
        return [
            'Overview' => [
                'icon' => 'heroicon-o-home',
                'component' => 'member.tabs.overview-tab',
            ],
            'Articles' => [
                'icon' => 'heroicon-o-newspaper',
                'component' => 'member.tabs.articles-tab',
                'badge' => $this->record->articles()->count(),
            ],
            'Classifieds' => [
                'icon' => 'heroicon-o-squares-2x2',
                'component' => 'member.tabs.classifieds-tab',
                'badge' => $this->record->classifieds()->count(),
            ],
            'Jobs' => [
                'icon' => 'heroicon-o-briefcase',
                'component' => 'member.tabs.jobs-tab',
                'badge' => $this->record->jobs()->count(),
            ],
            'Projects' => [
                'icon' => 'heroicon-o-building-office',
                'component' => 'member.tabs.projects-tab',
                'badge' => $this->record->projects()->count(),
            ],
            'Team' => [
                'icon' => 'heroicon-o-users',
                'component' => 'member.tabs.team-members-tab',
                'badge' => $this->record->team()->count(),
            ],
        ];
    }

    #[Computed]
    public function overflowTabs(): array
    {
        return [
            'Subscriptions' => [
                'icon' => 'heroicon-o-credit-card',
                'component' => 'member.tabs.subscriptions-tab',
                'badge' => $this->record->subscriptions()->count(),
            ],
            'Transactions' => [
                'icon' => 'heroicon-o-currency-dollar',
                'component' => 'member.tabs.transactions-tab',
                'badge' => $this->record->purchases()->count(),
            ],
            'Reviews' => [
                'icon' => 'heroicon-o-star',
                'component' => 'member.tabs.reviews-tab',
                'badge' => $this->record->reviews()->count(),
            ],
            // 'Activity' => [
            //     'icon' => 'heroicon-o-clock',
            //     'component' => null, // To be implemented
            // ],
            'Invoices' => [
                'icon' => 'heroicon-o-receipt-percent',
                'component' => 'member.tabs.invoices-tab',
                'badge' => $this->record->payments()->count(),
            ],
        ];
    }
}
