<?php

namespace App\Filament\Resources\Member\Trait;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\CheckboxList;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Get;
use App\Enums\UserProfileStatus;
use App\Forms\Components\TinyMceEditor;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Hidden;
use App\Enums\ProfessionalCredentionals;
use App\Enums\DayOfWeek;
use App\Services\User\BusinessHourService;
use App\Services\User\MemberCategoryService;
use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Forms\Components\RichEditor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

trait MemberFormTrait
{
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 12,
                    ])
                    ->schema([
                        // Main Content Column
                        Group::make()
                            ->columnSpan(['lg' => 8])
                            ->schema([
                                self::personalInformationSection(),
                                self::companyInformationSection(),
                                self::companyDescriptionSection(),
                                self::memberCategoriesSection(),
                                self::locationSection(),
                                self::professionalCredentialsSection(),
                                self::awardsAndRecognitionsSection(),
                                self::operatingHoursSection(),
                                self::profileTagsSection(),
                                self::socialMediaSection(),
                                self::gallerySection(),
                                self::faqsSection(),
                                self::securitySection(),
                            ]),

                        // Sidebar Column
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                self::profileSettingsSection(),
                            ]),
                    ]),
            ]);
    }

    protected static function personalInformationSection(): Section
    {
        return Section::make('Personal Information')
            ->description('Basic member information')
            ->collapsible()
            ->schema([
                TextInput::make('name')
                    ->label('Full Name')
                    ->hidden()
                    ->dehydrated()
                    ->live()
                    ->afterStateUpdated(function (string $operation, $state, Get $get, Forms\Set $set): void {
                        $fullName = $get('profile.first_name') . ' ' . $get('profile.last_name');
                        $set('name', $fullName);
                    }),
                Grid::make(2)->schema([
                    TextInput::make('profile.first_name')
                        ->label('First Name')
                        ->required()
                        ->live()
                        ->maxLength(255),
                    TextInput::make('profile.last_name')
                        ->label('Last Name')
                        ->required()
                        ->maxLength(255),
                ]),
                Grid::make(2)->schema([
                    TextInput::make('email')
                        ->label('Email')
                        ->email()
                        ->required()
                        ->maxLength(255),
                    TextInput::make('profile.phone')
                        ->label('Phone')
                        ->tel()
                        ->mask('(999) 999-9999')
                        ->required()
                        ->maxLength(255),
                ]),
                Grid::make(2)->schema([
                    TextInput::make('profile.mobile')
                        ->label('Mobile')
                        ->tel()
                        ->mask('(999) 999-9999')
                        ->required()
                        ->maxLength(255),
                    TextInput::make('profile.fax')
                        ->label('Fax')
                        ->mask('(999) 999-9999')
                        ->tel()
                        ->maxLength(255),
                ]),
            ]);
    }

    protected static function companyInformationSection(): Section
    {
        return Section::make('Basic Company Information')
            ->description('General company details')
            ->collapsible()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('profile.company_name')
                        ->label('Company Name')
                        ->required()
                        ->maxLength(255),
                    TextInput::make('profile.website')
                        ->label('Company Website')
                        ->url()
                        ->required()
                        ->maxLength(255),
                ]),
                Grid::make(2)->schema([
                    TextInput::make('profile.year_established')
                        ->label('Year Established')
                        ->required()
                        ->numeric(),
                    Select::make('profile.meta.business_timezone')
                        ->label('Business Timezone')
                        ->options(collect(timezone_identifiers_list())->mapWithKeys(fn($timezone) => [$timezone => $timezone]))
                        ->searchable()
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->profile && empty($state)) {
                                $timezone = $record->profile->getMeta('business_timezone');
                                if ($timezone) {
                                    $component->state($timezone);
                                }
                            }
                        })
                        ->dehydrated(false)
                        ->saveRelationshipsUsing(function ($record, $state) {
                            if ($record->profile) {
                                $record->profile->setMeta('business_timezone', $state);
                                $record->profile->save();
                            }
                        }),
                ]),
                Grid::make(1)->schema([
                    Textarea::make('profile.slogan')
                        ->label('Company Slogan')
                        ->required()
                        ->rows(2),
                ]),
            ]);
    }

    protected static function companyDescriptionSection(): Section
    {
        return Section::make('Company Description')
            ->description('Detailed information about the company')
            ->collapsible()
            ->schema([
                TinyMceEditor::make('profile.about')
                    ->label('About Company')
                    ->plugins('lists link autolink autoresize')
                    ->toolbar('undo redo | formatselect | h2 h3 h4 | bold italic underline | bullist numlist')
                    ->menubar(false)
                    ->branding(false)
                    ->promotion(false)
                    ->height(300)
                    ->required(),
            ]);
    }

    protected static function memberCategoriesSection(): Section
    {
        return Section::make('Business Categories')
            ->description('Select the categories your business belongs to. You can select multiple categories at any level.')
            ->collapsible()
            ->schema([
                // Parent categories selection
                Select::make('parent_categories')
                    ->label('Parent Categories')
                    ->multiple()
                    ->searchable()
                    ->preload()
                    ->required()
                    ->options(function () {
                        $categoryService = app(MemberCategoryService::class);
                        return $categoryService->getParentCategories()->pluck('category_name', 'id')->toArray();
                    })
                    ->helperText('Select one or more parent categories. Child categories will appear below after selection.')
                    ->reactive()
                    ->afterStateUpdated(function (callable $set, callable $get, $state) {
                        // Clear child category selections when parent changes
                        foreach ($get('parent_categories') ?? [] as $parentId) {
                            $set("child_categories_$parentId", []);
                            // Clear second level for this parent's children
                            $categoryService = app(MemberCategoryService::class);
                            $children = $categoryService->getChildCategories($parentId);
                            foreach ($children as $child) {
                                $set("second_level_categories_{$child->id}", []);
                            }
                        }

                        // Update all selected categories
                        static::updateAllSelectedCategories($set, $get);
                    })
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->memberCategories()->exists()) {
                            $userCategories = $record->memberCategories()->get();
                            $parentCategories = $userCategories->whereNull('parent_id')->pluck('id')->toArray();
                            $component->state($parentCategories);
                        }
                    })
                    ->columnSpanFull(),

                // Hidden field to store all selected categories
                Hidden::make('all_selected_categories')
                    ->default([])
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->memberCategories()->exists()) {
                            $userCategories = $record->memberCategories()->get();
                            $allSelected = $userCategories->pluck('id')->toArray();
                            $component->state($allSelected);
                        }
                    })
                    ->dehydrated(),

                // Dynamic sections for each parent category
                ...static::generateCategorySubSections(),
            ])
            ->dehydrated(false)
            ->saveRelationshipsUsing(function ($record, $state) {
                // Get all selected categories from the hidden field
                $allCategoryIds = $state['all_selected_categories'] ?? [];

                // Also collect from individual fields as backup
                $collectedIds = [];

                // Add parent categories
                $collectedIds = array_merge($collectedIds, $state['parent_categories'] ?? []);

                // Add child and second level categories
                foreach ($state as $key => $value) {
                    if (str_starts_with($key, 'child_categories_') || str_starts_with($key, 'second_level_categories_')) {
                        $collectedIds = array_merge($collectedIds, (array) $value);
                    }
                }

                // Use the collected IDs if all_selected_categories is empty
                if (empty($allCategoryIds) && !empty($collectedIds)) {
                    $allCategoryIds = $collectedIds;
                }

                // Remove duplicates and sync to relationship
                $allCategoryIds = array_unique($allCategoryIds);
                $record->memberCategories()->sync($allCategoryIds);
            });
    }

    protected static function generateCategorySubSections(): array
    {
        $categoryService = app(MemberCategoryService::class);
        $parentCategories = $categoryService->getParentCategories();
        $sections = [];

        foreach ($parentCategories as $parent) {
            $children = $categoryService->getChildCategories($parent->id);

            if ($children->isNotEmpty()) {
                // Section for this parent category
                $childComponents = [
                    CheckboxList::make("child_categories_{$parent->id}")
                        ->label('')
                        ->options($children->pluck('category_name', 'id')->toArray())
                        ->reactive()
                        ->afterStateUpdated(function (callable $set, callable $get, $state) use ($parent, $categoryService) {
                            // Update all_selected_categories when child selection changes
                            static::updateAllSelectedCategories($set, $get);

                            // Clear second level selections for unselected children
                            $children = $categoryService->getChildCategories($parent->id);
                            foreach ($children as $child) {
                                if (!in_array($child->id, $state ?? [])) {
                                    $set("second_level_categories_{$child->id}", []);
                                }
                            }
                        })
                        ->afterStateHydrated(function ($component, $state, $record) use ($parent) {
                            if ($record && $record->memberCategories()->exists()) {
                                $userCategories = $record->memberCategories()->get();
                                $childCategories = $userCategories->filter(function ($category) use ($parent) {
                                    return $category->parent_id == $parent->id;
                                })->pluck('id')->toArray();
                                $component->state($childCategories);
                            }
                        })
                        ->columnSpanFull(),
                ];

                // Add second level checkboxes for each child
                foreach ($children as $child) {
                    $secondLevel = $categoryService->getChildCategories($child->id);

                    if ($secondLevel->isNotEmpty()) {
                        $childComponents[] = CheckboxList::make("second_level_categories_{$child->id}")
                            ->label($child->category_name)
                            ->options($secondLevel->pluck('category_name', 'id')->toArray())
                            ->visible(fn(callable $get) => in_array($child->id, $get("child_categories_{$parent->id}") ?? []))
                            ->reactive()
                            ->afterStateUpdated(function (callable $set, callable $get, $state) {
                                // Update all_selected_categories when second level selection changes
                                static::updateAllSelectedCategories($set, $get);
                            })
                            ->afterStateHydrated(function ($component, $state, $record) use ($child) {
                                if ($record && $record->memberCategories()->exists()) {
                                    $userCategories = $record->memberCategories()->get();
                                    $secondLevelCategories = $userCategories->filter(function ($category) use ($child) {
                                        return $category->parent_id == $child->id;
                                    })->pluck('id')->toArray();
                                    $component->state($secondLevelCategories);
                                }
                            })
                            ->columnSpanFull();
                    }
                }

                $sections[] = Section::make($parent->category_name)
                    ->schema($childComponents)
                    ->visible(fn(callable $get) => in_array($parent->id, $get('parent_categories') ?? []))
                    ->collapsible()
                    ->collapsed(false);
            }
        }

        return $sections;
    }

    protected static function updateAllSelectedCategories(callable $set, callable $get): void
    {
        $allSelected = [];

        // Add parent categories
        $allSelected = array_merge($allSelected, $get('parent_categories') ?? []);

        // Add child and second level categories from all fields
        $allState = $get('/');
        foreach ($allState as $key => $value) {
            if (str_starts_with($key, 'child_categories_') || str_starts_with($key, 'second_level_categories_')) {
                $allSelected = array_merge($allSelected, (array) $value);
            }
        }

        $set('all_selected_categories', array_unique($allSelected));
    }

    protected static function locationSection(): Section
    {
        return Section::make('Location Information')
            ->description('Add your primary business location')
            ->collapsible()
            ->schema([
                // Contact Information Fields (above the map)
                Grid::make(2)->schema([
                    TextInput::make('location_name')
                        ->label('Location Name')
                        ->required()
                        ->maxLength(255)
                        ->placeholder('e.g. Main Office')
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->locations()->exists()) {
                                $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                if ($primaryLocation) {
                                    $component->state($primaryLocation->location_name);
                                }
                            }
                        }),
                    TextInput::make('location_contact_person_name')
                        ->label('Contact Person')
                        ->required()
                        ->maxLength(255)
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->locations()->exists()) {
                                $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                if ($primaryLocation) {
                                    $component->state($primaryLocation->contact_person_name);
                                }
                            }
                        }),
                ]),
                Grid::make(3)->schema([
                    TextInput::make('location_phone')
                        ->label('Phone')
                        ->tel()
                        ->mask('(999) 999-9999')
                        ->required()
                        ->maxLength(255)
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->locations()->exists()) {
                                $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                if ($primaryLocation) {
                                    $component->state($primaryLocation->phone);
                                }
                            }
                        }),
                    TextInput::make('location_email')
                        ->label('Email')
                        ->email()
                        ->maxLength(255)
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->locations()->exists()) {
                                $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                if ($primaryLocation) {
                                    $component->state($primaryLocation->email);
                                }
                            }
                        }),
                    TextInput::make('location_fax')
                        ->label('Fax')
                        ->tel()
                        ->mask('(999) 999-9999')
                        ->maxLength(255)
                        ->afterStateHydrated(function ($component, $state, $record) {
                            if ($record && $record->locations()->exists()) {
                                $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                if ($primaryLocation) {
                                    $component->state($primaryLocation->fax);
                                }
                            }
                        }),
                ]),

                // Location field (like in Classified)
                TextInput::make('location')
                    ->columnSpanFull()
                    ->required()
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->locations()->exists()) {
                            $primaryLocation = $record->locations()->where('primary_location', true)->first();
                            if ($primaryLocation) {
                                // Use city name or location name as the location field value
                                $locationValue = $primaryLocation->city?->city_name ?: $primaryLocation->location_name;
                                $component->state($locationValue);
                            }
                        }
                    }),

                // Address section with Map and Form fields (like in Classified)
                Grid::make()
                    ->columns(2)
                    ->schema([
                        // Map Column
                        Group::make()
                            ->columnSpan(1)
                            ->schema([
                                Map::make('map_address')
                                    ->mapControls([
                                        'mapTypeControl'    => true,
                                        'scaleControl'      => true,
                                        'streetViewControl' => true,
                                        'rotateControl'     => true,
                                        'fullscreenControl' => true,
                                        'searchBoxControl'  => false,
                                        'zoomControl'       => true,
                                    ])
                                    ->defaultZoom(13)
                                    ->defaultLocation([0, 0])
                                    ->draggable()
                                    ->clickable()
                                    ->autocomplete('location')
                                    ->autocompleteReverse()
                                    ->reverseGeocode([
                                        'address' => '%n %S %D %L %z',
                                        'postal_code' => '%z',
                                        'city_name' => '%L',
                                        'state_name' => '%A1',
                                        'country_name' => '%C',
                                        'country_code' => '%c',
                                    ])
                                    ->afterStateUpdated(function ($state, Forms\Set $set, Forms\Get $get) {
                                        if (isset($state['lat']) && isset($state['lng'])) {
                                            $set('latitude', $state['lat']);
                                            $set('longitude', $state['lng']);

                                            if (isset($state['address'])) {
                                                $set('address', $state['address']);
                                            }
                                            if (isset($state['postal_code'])) {
                                                $set('postal_code', $state['postal_code']);
                                            }
                                            if (isset($state['country_name'])) {
                                                $set('country_name', $state['country_name']);
                                            }
                                            if (isset($state['country_code'])) {
                                                $set('country_code', $state['country_code']);
                                            }
                                            if (isset($state['state_name'])) {
                                                $set('state_name', $state['state_name']);
                                            }
                                            // Set city field with more aggressive retry
                                            if (isset($state['city_name']) && !empty($state['city_name'])) {
                                                $cityName = $state['city_name'];
                                                $set('city_name', $cityName);

                                                // Immediate retry - helps with timing issues
                                                $set('city_name', $cityName);
                                            }
                                        }
                                    })
                                    ->afterStateHydrated(function ($component, $state, $record) {
                                        if ($record && $record->locations()->exists()) {
                                            $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                            if ($primaryLocation && $primaryLocation->latitude && $primaryLocation->longitude) {
                                                $component->state([
                                                    'lat' => $primaryLocation->latitude,
                                                    'lng' => $primaryLocation->longitude,
                                                    'address' => $primaryLocation->map_address ?: $primaryLocation->address_line_1
                                                ]);
                                            }
                                        }
                                    }),
                            ]),

                        // Address Fields Column (like in Classified)
                        Group::make()
                            ->columnSpan(1)
                            ->schema([
                                Grid::make()
                                    ->columns(2)
                                    ->schema([
                                        TextInput::make('address')
                                            ->label('Street Address')
                                            ->required()
                                            ->columnSpanFull()
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation) {
                                                        $component->state($primaryLocation->address_line_1);
                                                    }
                                                }
                                            }),
                                        TextInput::make('latitude')
                                            ->numeric()
                                            ->columnSpan(1)
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation) {
                                                        $component->state($primaryLocation->latitude);
                                                    }
                                                }
                                            }),
                                        TextInput::make('longitude')
                                            ->numeric()
                                            ->columnSpan(1)
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation) {
                                                        $component->state($primaryLocation->longitude);
                                                    }
                                                }
                                            }),
                                        TextInput::make('country_name')
                                            ->label('Country')
                                            ->columnSpan(1)
                                            ->readOnly()
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation && $primaryLocation->country) {
                                                        $component->state($primaryLocation->country->country_name);
                                                    }
                                                }
                                            }),
                                        TextInput::make('state_name')
                                            ->label('State')
                                            ->columnSpan(1)
                                            ->readOnly()
                                            ->live()
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation && $primaryLocation->state) {
                                                        $component->state($primaryLocation->state->state_name);
                                                    }
                                                }
                                            }),
                                        TextInput::make('city_name')
                                            ->label('City')
                                            ->columnSpan(1)
                                            ->readOnly()
                                            ->live()
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation && $primaryLocation->city) {
                                                        $component->state($primaryLocation->city->city_name);
                                                    }
                                                }
                                            }),
                                        TextInput::make('postal_code')
                                            ->label('Postal Code')
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if ($record && $record->locations()->exists()) {
                                                    $primaryLocation = $record->locations()->where('primary_location', true)->first();
                                                    if ($primaryLocation) {
                                                        $component->state($primaryLocation->postal_code);
                                                    }
                                                }
                                            }),
                                        Hidden::make('country_code'),
                                        Hidden::make('country_id'),
                                        Hidden::make('state_id'),
                                        Hidden::make('city_id'),
                                    ]),
                            ]),
                    ]),
                // Service Areas Section
                TagsInput::make('service_areas')
                    ->label('Service Areas')
                    ->placeholder('Add a service area and press Enter')
                    ->helperText('Enter service areas one at a time and press Enter. Areas will be linked to this location.')
                    ->separator(',')
                    ->reorderable()
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->locations()->exists()) {
                            $primaryLocation = $record->locations()->where('primary_location', true)->first();
                            if ($primaryLocation) {
                                $serviceAreas = $primaryLocation->areas->pluck('name')->toArray();
                                $component->state($serviceAreas);
                            }
                        }
                    }),
            ])
            ->dehydrated(false)
            ->saveRelationshipsUsing(function ($record, $state) {
                try {
                    // Extract location data from individual fields
                    $locationData = [
                        'user_id' => $record->id,
                        'location_name' => $state['location_name'] ?? '',
                        'contact_person_name' => $state['location_contact_person_name'] ?? '',
                        'phone' => $state['location_phone'] ?? '',
                        'email' => $state['location_email'] ?? '',
                        'fax' => $state['location_fax'] ?? null,
                        'address_line_1' => $state['address'] ?? '',
                        'address_line_2' => null,
                        'postal_code' => $state['postal_code'] ?? null,
                        'latitude' => $state['latitude'] ? (float)$state['latitude'] : null,
                        'longitude' => $state['longitude'] ? (float)$state['longitude'] : null,
                        'map_address' => $state['address'] ?? '',
                        'primary_location' => true,
                        'is_active' => true,
                        'created_by' => Auth::id(), // Add created_by for UserLocation
                    ];

                    // Validate required fields - check individual field names properly
                    $requiredFields = [
                        'location_name' => 'Location Name',
                        'address' => 'Street Address'
                    ];

                    $missingFields = [];
                    foreach ($requiredFields as $field => $label) {
                        if (empty($state[$field])) {
                            $missingFields[] = $label;
                        }
                    }

                    if (!empty($missingFields)) {
                        \Filament\Notifications\Notification::make()
                            ->danger()
                            ->title('Location Save Failed')
                            ->body('Missing required fields: ' . implode(', ', $missingFields))
                            ->send();
                        return;
                    }

                    // Handle country
                    $countryId = null;
                    if (!empty($state['country_name'])) {
                        $countryData = [
                            'country_name' => $state['country_name'],
                            'created_by' => Auth::id(),
                        ];

                        // Add country_code if available from Google Maps
                        if (!empty($state['country_code'])) {
                            $countryData['country_code'] = strtoupper($state['country_code']);
                        } else {
                            // Fallback: generate country code from country name (first 2 letters)
                            $countryData['country_code'] = strtoupper(substr($state['country_name'], 0, 2));
                        }

                        $country = \App\Models\Location\Country::firstOrCreate(
                            ['country_name' => $state['country_name']],
                            $countryData
                        );
                        if ($country) {
                            $countryId = $country->id;
                            $locationData['country_id'] = $countryId;
                        }
                    }

                    // Handle state
                    $stateId = null;
                    if (!empty($state['state_name'])) {
                        $stateData = [
                            'state_name' => $state['state_name'],
                            'created_by' => Auth::id(),
                        ];
                        if ($countryId) {
                            $stateData['country_id'] = $countryId;
                        }
                        $stateModel = \App\Models\Location\State::firstOrCreate(
                            ['state_name' => $state['state_name'], 'country_id' => $countryId],
                            $stateData
                        );
                        if ($stateModel) {
                            $stateId = $stateModel->id;
                            $locationData['state_id'] = $stateId;
                        }
                    }

                    // Handle city
                    if (!empty($state['city_name'])) {
                        $cityData = [
                            'city_name' => $state['city_name'],
                            'created_by' => Auth::id(),
                        ];
                        if ($stateId) $cityData['state_id'] = $stateId;
                        if ($countryId) $cityData['country_id'] = $countryId;

                        $city = \App\Models\Location\City::firstOrCreate(
                            ['city_name' => $state['city_name'], 'state_id' => $stateId, 'country_id' => $countryId],
                            $cityData
                        );
                        if ($city) {
                            $locationData['city_id'] = $city->id;
                        }
                    }

                    // Update or create the primary location
                    $location = $record->locations()->updateOrCreate(
                        ['user_id' => $record->id, 'primary_location' => true],
                        $locationData
                    );

                    if ($location) {
                        // Handle service areas if provided
                        if (isset($state['service_areas']) && is_array($state['service_areas']) && !empty($state['service_areas']) && $location->city_id) {
                            $areaService = app(\App\Services\Location\AreaService::class);
                            $areaService->syncAreasToLocation($location, $state['service_areas']);
                        } elseif ($location->city_id) {
                            // Clear service areas if none provided
                            $location->areas()->detach();
                        }

                        \Filament\Notifications\Notification::make()
                            ->success()
                            ->title('Location Saved')
                            ->body('Location information has been saved successfully.')
                            ->send();
                    }
                } catch (\Exception $e) {
                    // Log the error and show notification
                    Log::error('Location save failed for user ' . $record->id . ': ' . $e->getMessage(), [
                        'exception' => $e,
                        'state' => $state,
                        'user_id' => $record->id
                    ]);

                    \Filament\Notifications\Notification::make()
                        ->danger()
                        ->title('Location Save Failed')
                        ->body('An error occurred while saving location: ' . $e->getMessage())
                        ->send();
                }
            });
    }

    protected static function professionalCredentialsSection(): Section
    {
        return Section::make('Professional Credentials')
            ->description('Professional credentials and certifications held by the company')
            ->collapsible()
            ->collapsed()
            ->schema([
                Repeater::make('profile.credentials')
                    ->label('Professional Credentials')
                    ->schema([
                        Select::make('credential_type')
                            ->label('Credential Type')
                            ->options(\App\Enums\ProfessionalCredentionals::class)
                            ->required(),
                        TextInput::make('credential_value')
                            ->label('Credential/Certification')
                            ->required()
                            ->maxLength(100),
                    ])
                    ->itemLabel(
                        fn(array $state): ?string =>
                        isset($state['credential_type']) ? $state['credential_type'] : null
                    )
                    ->collapsible()
                    ->collapsed()
                    ->defaultItems(0)
                    ->addActionLabel('Add Credential')
                    ->reorderableWithButtons()
                    ->maxItems(4)
                    ->columns(2),
            ]);
    }

    protected static function awardsAndRecognitionsSection(): Section
    {
        return Section::make('Awards & Recognitions')
            ->description('Awards and recognitions received by the company')
            ->collapsible()
            ->collapsed()
            ->schema([
                Repeater::make('profile.meta.awards_recognitions')
                    ->label('Awards & Recognitions')
                    ->schema([
                        TextInput::make('title')
                            ->label('Award Title')
                            ->required()
                            ->maxLength(100),
                        TextInput::make('year')
                            ->label('Year Received')
                            ->required()
                            ->numeric()
                            ->minValue(1800)
                            ->maxValue(date('Y'))
                    ])
                    ->itemLabel(
                        fn(array $state): ?string =>
                        isset($state['title']) ? $state['title'] . ' (' . ($state['year'] ?? '') . ')' : null
                    )
                    ->columns(2)
                    ->collapsible()
                    ->collapsed()
                    ->defaultItems(0)
                    ->addActionLabel('Add Award')
                    ->reorderableWithButtons()
                    ->maxItems(4)
                    ->afterStateHydrated(function (Repeater $component, $state, $record) {
                        if ($record && $record->profile && empty($state)) {
                            $awards = $record->profile->getMeta('awards_recognitions');
                            if (!empty($awards)) {
                                $component->state($awards);
                            }
                        }
                    })
                    ->dehydrated(false)
                    ->saveRelationshipsUsing(function ($record, $state) {
                        if ($record->profile) {
                            $record->profile->setMeta('awards_recognitions', $state);
                            $record->profile->save();
                        }
                    }),
            ]);
    }

    protected static function operatingHoursSection(): Section
    {
        return Section::make('Operating Hours')
            ->description('Configure business operating hours')
            ->collapsible()
            ->collapsed()
            ->schema([
                // Business Hours by Day - Fixed 7 days
                Repeater::make('businessHours')
                    ->label('Business Hours')
                    ->relationship('businessHours')
                    ->schema([
                        Grid::make(5)->schema([
                            Select::make('day_of_week')
                                ->label('Day')
                                ->options(collect(DayOfWeek::cases())->mapWithKeys(fn($day) => [$day->value => $day->getLabel()]))
                                ->required()
                                ->disabled()
                                ->dehydrated()
                                ->columnSpan(1),
                            Checkbox::make('is_open')
                                ->label('Open')
                                ->default(true)
                                ->live()
                                ->columnSpan(1),
                            TextInput::make('hours_label')
                                ->label('Hours Label')
                                ->placeholder('e.g. Regular Hours, Appointment Only')
                                ->maxLength(30)
                                ->columnSpan(1),
                            TextInput::make('open_time')
                                ->label('Open Time')
                                ->type('time')
                                ->required()
                                ->columnSpan(1)
                                ->visible(fn(Get $get) => $get('is_open')),
                            TextInput::make('close_time')
                                ->label('Close Time')
                                ->type('time')
                                ->required()
                                ->columnSpan(1)
                                ->visible(fn(Get $get) => $get('is_open')),
                        ])
                    ])
                    ->deletable(false)
                    ->addable(false)
                    ->reorderable(false)
                    ->itemLabel(
                        fn(array $state): ?string =>
                        isset($state['day_of_week'])
                            ? (DayOfWeek::tryFrom($state['day_of_week'])?->getLabel() ?? 'Day' . ': ' .
                                ($state['is_open']
                                    ? ($state['open_time'] ?? '00:00') . ' - ' . ($state['close_time'] ?? '00:00')
                                    : 'Closed') .
                                (isset($state['hours_label']) && $state['hours_label'] ? ' (' . $state['hours_label'] . ')' : ''))
                            : null
                    )
                    ->collapsible()
                    ->collapsed()
                    ->defaultItems(0)
                    ->columns(1)
                    ->afterStateHydrated(function (Repeater $component, ?array $state, $record) {
                        // If no business hours exist, create default ones
                        if (empty($state) && $record) {
                            $businessHourService = app(BusinessHourService::class);
                            if ($record->businessHours()->count() === 0) {
                                $businessHourService->createDefaultBusinessHours($record);
                                // Refresh the record to get the newly created business hours
                                $record->refresh();
                            }
                        }

                        // Ensure all 7 days are represented
                        if ($record && $record->businessHours()->count() < 7) {
                            $existingDays = $record->businessHours->pluck('day_of_week')->map(fn($day) => $day->value)->toArray();
                            $allDays = collect(DayOfWeek::cases())->map(fn($day) => $day->value)->toArray();
                            $missingDays = array_diff($allDays, $existingDays);

                            foreach ($missingDays as $day) {
                                $record->businessHours()->create([
                                    'day_of_week' => $day,
                                    'is_open' => true,
                                    'hours_label' => '',
                                    'open_time' => '09:00',
                                    'close_time' => '17:00',
                                ]);
                            }
                            $record->refresh();
                        }
                    })
                    ->default(function () {
                        // For create mode, return default business hours structure
                        $defaultBusinessHours = [];
                        foreach (DayOfWeek::cases() as $day) {
                            $defaultBusinessHours[] = [
                                'day_of_week' => $day->value,
                                'is_open' => in_array($day, [DayOfWeek::SATURDAY, DayOfWeek::SUNDAY]) ? false : true,
                                'hours_label' => in_array($day, [DayOfWeek::SATURDAY, DayOfWeek::SUNDAY]) ? null : '',
                                'open_time' => in_array($day, [DayOfWeek::SATURDAY, DayOfWeek::SUNDAY]) ? null : '09:00',
                                'close_time' => in_array($day, [DayOfWeek::SATURDAY, DayOfWeek::SUNDAY]) ? null : '17:00',
                            ];
                        }
                        return $defaultBusinessHours;
                    })
                    ->mutateRelationshipDataBeforeCreateUsing(function (array $data): array {
                        if (!isset($data['is_open'])) {
                            $data['is_open'] = true;
                        }
                        if (!$data['is_open']) {
                            $data['open_time'] = null;
                            $data['close_time'] = null;
                        }
                        return $data;
                    })
                    ->mutateRelationshipDataBeforeSaveUsing(function (array $data): array {
                        if (!isset($data['is_open'])) {
                            $data['is_open'] = true;
                        }
                        if (!$data['is_open']) {
                            $data['open_time'] = null;
                            $data['close_time'] = null;
                        }
                        return $data;
                    }),

                // Seasonal Hours (keep as meta data)
                Repeater::make('profile.meta.seasonal_hours')
                    ->label('Seasonal Hours')
                    ->schema([
                        Checkbox::make('is_open')
                            ->label('Open')
                            ->default(true)
                            ->live(),
                        Grid::make(1)->schema([
                            TextInput::make('season_name')
                                ->label('Season Name')
                                ->required()
                                ->maxLength(20)
                                ->placeholder('e.g. Summer Hours'),
                        ]),
                        Grid::make(2)->schema([
                            TextInput::make('start_date')
                                ->label('Start Date')
                                ->required()
                                ->type('date'),
                            TextInput::make('end_date')
                                ->label('End Date')
                                ->required()
                                ->type('date'),
                        ]),
                        Repeater::make('seasonal_days_with_hours')
                            ->label('Days & Hours')
                            ->schema([
                                Grid::make(1)->schema([
                                    TextInput::make('day')
                                        ->label('Day')
                                        ->required(),
                                ]),
                                Grid::make(2)->schema([
                                    TextInput::make('start_time')
                                        ->label('Start Time')
                                        ->required()
                                        ->type('time'),
                                    TextInput::make('end_time')
                                        ->label('End Time')
                                        ->required()
                                        ->type('time'),
                                ]),
                            ])

                            ->itemLabel(
                                fn(array $state): ?string =>
                                isset($state['day']) ? $state['day'] . ': ' . ($state['start_time'] ?? '') . ' - ' . ($state['end_time'] ?? '') : null
                            )
                            ->collapsible()
                            ->defaultItems(0)
                            ->addActionLabel('Add Day')
                            ->reorderableWithButtons()
                            ->maxItems(7)
                            ->columns(3)
                            ->visible(fn(Get $get) => $get('is_open')),
                    ])
                    ->itemLabel(
                        fn(array $state): ?string =>
                        isset($state['season_name'], $state['start_date'], $state['end_date'])
                            ? ($state['season_name'] . ': ' . $state['start_date'] . ' to ' . $state['end_date'])
                            : null
                    )
                    ->collapsible()
                    ->collapsed()
                    ->defaultItems(0)
                    ->addActionLabel('Add Seasonal Hours')
                    ->reorderableWithButtons()
                    ->maxItems(7)
                    ->columns(1)
                    ->afterStateHydrated(function (Repeater $component, $state, $record) {
                        if ($record && $record->profile && empty($state)) {
                            $seasonalHours = $record->profile->getMeta('seasonal_hours');
                            if (!empty($seasonalHours)) {
                                $component->state($seasonalHours);
                            }
                        }
                    })
                    ->dehydrated(false)
                    ->saveRelationshipsUsing(function ($record, $state) {
                        if ($record->profile) {
                            $record->profile->setMeta('seasonal_hours', $state);
                            $record->profile->save();
                        }
                    }),



            ]);
    }

    protected static function profileTagsSection(): Section
    {
        return Section::make('Profile Tags')
            ->description('Add up to 3 tags for your profile')
            ->collapsible()
            ->collapsed()
            ->schema([
                // Profile Tags (meta)
                // TagsInput::make('profile.meta.profile_tags')
                //     ->rules(['max:3'])
                //     ->label('Profile Tags')
                //     ->placeholder('Add a tag and press Enter')
                //     ->helperText('User can add max 3 tags. Each tag length should not be more then 20 chars. ')
                //     ->columnSpanFull()
                //     ->disabled(fn(Get $get) => $get('profile.meta.profile_tags') && count($get('profile.meta.profile_tags')) >= 3)
                //     ->nestedRecursiveRules([
                //         'min:3',
                //         'max:20',
                //     ])
                //     ->reorderable()
                // TagsInput::make('profile.meta.profile_tags')
                //     ->label('Profile Tags')
                //     ->placeholder('Add a tag and press Enter')
                //     ->helperText('User can add max 3 tags. Each tag length should not be more than 20 chars.')
                //     ->columnSpanFull()
                //     ->reactive()
                //     ->disabled(fn(Get $get) => count($get('profile.meta.profile_tags') ?? []) >= 3)
                //     ->rules(['array', 'max:3'])
                //     ->nestedRecursiveRules([
                //         'min:3', // each tag min 3 chars
                //         'max:20', // each tag max 20 chars
                //     ])
                //     ->reorderable()
                TagsInput::make('profile.meta.profile_tags')
                    ->label('Profile Tags')
                    ->placeholder('Add a tag and press Enter')
                    ->helperText('User can add max 3 tags. Each tag length should not be more than 20 chars.')
                    ->columnSpanFull()
                    ->reactive()
                    ->rules(['array', 'max:3']) // enforce max tags in validation
                    ->nestedRecursiveRules([
                        'min:3', // each tag min 3 chars
                        'max:20', // each tag max 20 chars
                    ])
                    ->afterStateUpdated(function ($state, callable $set) {
                        // Validate each tag length and filter out invalid ones
                        // if ($state) {
                        //     $state = array_filter($state, function($tag) {
                        //         return strlen($tag) <= 20;
                        //     });
                        //     $set('profile.meta.profile_tags', $state);
                        // }
                        if (count($state ?? []) > 3) {
                            // keep only first 3 tags
                            $set('profile.meta.profile_tags', array_slice($state, 0, 3));
                        }
                    })
                    ->reorderable()
            ]);
    }

    protected static function socialMediaSection(): Section
    {
        return Section::make('Social Media Profiles')
            ->description('Connect your social media accounts')
            ->collapsible()
            ->collapsed()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('profile.social_media_links.facebook')
                        ->label('Facebook')
                        ->url()
                        ->prefixIcon('fab-facebook'),
                    TextInput::make('profile.social_media_links.instagram')
                        ->label('Instagram')
                        ->url()
                        ->prefixIcon('fab-instagram'),
                    TextInput::make('profile.social_media_links.twitter')
                        ->label('Twitter')
                        ->url()
                        ->prefixIcon('fab-twitter'),
                    TextInput::make('profile.social_media_links.linkedin')
                        ->label('LinkedIn')
                        ->url()
                        ->prefixIcon('fab-linkedin'),
                    TextInput::make('profile.social_media_links.youtube')
                        ->label('Youtube')
                        ->url()
                        ->prefixIcon('fab-youtube'),
                    TextInput::make('profile.social_media_links.tiktok')
                        ->label('TikTok')
                        ->url()
                        ->prefixIcon('fab-tiktok'),
                    TextInput::make('profile.social_media_links.pinterest')
                        ->label('Pinterest')
                        ->url()
                        ->prefixIcon('fab-pinterest'),
                ]),
            ]);
    }

    protected static function gallerySection(): Section
    {
        return Section::make('Gallery Images')
            ->description('Upload up to 10 images for your gallery')
            ->collapsible()
            ->collapsed()
            ->schema([
                SpatieMediaLibraryFileUpload::make('gallery')
                    ->label('Gallery Images')
                    ->collection('gallery')
                    ->multiple()
                    ->maxFiles(10)
                    ->conversion('medium')
                    ->reorderable()
                    ->extraAttributes([
                        'class' => 'upload-media-gallery',
                    ]),
            ]);
    }

    protected static function faqsSection(): Section
    {
        return Section::make('FAQs')
            ->description('Add FAQs for your profile')
            ->collapsible()
            ->collapsed()
            ->schema([
                Repeater::make('profile.meta.faqs')
                    ->label('FAQs')
                    ->schema([
                        TextInput::make('question')
                            ->label('Question'),
                        RichEditor::make('answer')
                            ->label('Answer'),
                    ])
                    ->reorderable(),
            ]);
    }

    protected static function securitySection(): Section
    {
        return Section::make('Security')
            ->description('Manage account password')
            ->collapsible()
            ->collapsed()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('password')
                        ->label('Password')
                        ->password()
                        ->live()
                        ->required(fn(string $operation): bool => $operation === 'create')
                        ->dehydrated(fn($state) => filled($state))
                        ->maxLength(255),
                    TextInput::make('password_confirmation')
                        ->label('Confirm Password')
                        ->password()
                        ->required(fn(Get $get): bool => filled($get('password')))
                        ->maxLength(255),
                ]),
                Checkbox::make('password_change_email')
                    ->label('Send password change notification email')
                    ->default(false)
                    ->hidden(fn(Get $get): bool => !filled($get('password'))),
            ]);
    }

    protected static function profileSettingsSection(): Section
    {
        return Section::make('Profile Settings')
            ->description('Manage profile visibility and categories')
            ->schema([
                SpatieMediaLibraryFileUpload::make('avatar')
                    ->label('Profile Picture')
                    ->collection('avatar')
                    ->imageEditor()
                    ->conversion('thumb')
                    ->imageResizeMode('cover')
                    ->circleCropper(),
                Select::make('profile.status')
                    ->label('Profile Status')
                    ->options(UserProfileStatus::class)
                    ->required()
                    ->default(UserProfileStatus::Active->value)
                    ->native(false),
                SpatieMediaLibraryFileUpload::make('logo')
                    ->label('Company Logo')
                    ->collection('logo')
                    ->imageEditor()
                    ->conversion('small'),
                SpatieMediaLibraryFileUpload::make('cover')
                    ->label('Cover Image')
                    ->collection('cover')
                    ->imageEditor()
                    ->conversion('medium'),
            ]);
    }
}
