<?php

namespace App\Filament\Resources\Member\Trait;

use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use App\Enums\UserProfileStatus;

trait MemberTableTrait
{
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                self::avatarColumn(),
                self::nameColumn(),
                self::emailColumn(),
                self::phoneColumn(),
                self::companyNameColumn(),
                self::statusColumn(),
                self::createdAtColumn(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                self::trashedFilter(),
            ])
            ->actions([
                self::actionGroup(),
            ])
            ->bulkActions([
                self::bulkActionGroup(),
            ]);
    }

    protected static function avatarColumn(): SpatieMediaLibraryImageColumn
    {
        return SpatieMediaLibraryImageColumn::make('avatar')
            ->collection('avatar')
            ->conversion('thumb')
            ->size(40)
            ->label('Avatar')
            ->square();
    }

    protected static function nameColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('name')
            ->searchable();
    }

    protected static function emailColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('email')
            ->searchable();
    }

    protected static function phoneColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('profile.phone')
            ->label('Phone')
            ->searchable();
    }

    protected static function companyNameColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('profile.company_name')
            ->label('Company Name')
            ->searchable();
    }

    protected static function statusColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('profile.status')
            ->label('Status')
            ->badge()
            ->formatStateUsing(fn($state) => $state instanceof UserProfileStatus ? $state->getLabel() : $state)
            ->color(fn($state) => $state instanceof UserProfileStatus ? $state->getColor() : 'gray')
            ->icon(fn($state) => $state instanceof UserProfileStatus ? $state->getIcon() : 'heroicon-o-question-mark-circle')
            ->searchable();
    }

    protected static function createdAtColumn(): Tables\Columns\TextColumn
    {
        return Tables\Columns\TextColumn::make('created_at')
            ->dateTime()
            ->sortable();
    }

    protected static function trashedFilter(): Tables\Filters\TrashedFilter
    {
        return Tables\Filters\TrashedFilter::make();
    }

    protected static function actionGroup(): Tables\Actions\ActionGroup
    {
        return Tables\Actions\ActionGroup::make([
            Tables\Actions\RestoreAction::make()
                ->visible(fn($record) => $record->trashed()),
            Tables\Actions\ForceDeleteAction::make()
                ->visible(fn($record) => $record->trashed()),
            Tables\Actions\Action::make('activate')
                ->label(
                    fn($record) =>
                    $record->profile?->status === UserProfileStatus::PendingApproval
                        ? 'Approve'
                        : 'Activate'
                )
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->requiresConfirmation()
                ->visible(fn($record) => !$record->trashed() && (!$record->profile || $record->profile->status !== UserProfileStatus::Active))
                ->action(function ($record) {
                    $record->profile()->update(['status' => UserProfileStatus::Active]);
                }),
            Tables\Actions\Action::make('services')
                ->label('Services')
                ->icon('heroicon-o-briefcase')
                ->color('info')
                ->url(fn($record) => route('filament.espadmin.resources.member.members.services', ['record' => $record]))
                ->visible(fn($record) => ! $record->trashed()),
            Tables\Actions\ViewAction::make()
                ->visible(fn($record) => ! $record->trashed()),
            Tables\Actions\EditAction::make()
                ->visible(fn($record) => ! $record->trashed()),
            Tables\Actions\Action::make('deactivate')
                ->label('Put On Hold')
                ->icon('heroicon-o-pause-circle')
                ->color('gray')
                ->requiresConfirmation()
                ->visible(fn($record) => !$record->trashed() && $record->profile && $record->profile->status === UserProfileStatus::Active)
                ->action(function ($record) {
                    $record->profile()->update(['status' => UserProfileStatus::OnHold]);
                }),
            Tables\Actions\Action::make('cancel')
                ->label('Cancel Account')
                ->icon('heroicon-o-x-circle')
                ->color('danger')
                ->requiresConfirmation()
                ->visible(fn($record) => !$record->trashed() && (!$record->profile || $record->profile->status !== UserProfileStatus::Cancelled))
                ->action(function ($record) {
                    $record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                }),
            Tables\Actions\DeleteAction::make()
                ->visible(fn($record) => ! $record->trashed())
                ->before(function ($record) {
                    $record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                }),
        ])
            ->tooltip('Actions');
    }

    protected static function bulkActionGroup(): Tables\Actions\BulkActionGroup
    {
        return Tables\Actions\BulkActionGroup::make([
            Tables\Actions\DeleteBulkAction::make()
                ->before(function ($records) {
                    $records->each(function ($record) {
                        $record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                    });
                }),
            Tables\Actions\ForceDeleteBulkAction::make(),
            Tables\Actions\RestoreBulkAction::make(),
            Tables\Actions\BulkAction::make('activateSelected')
                ->label('Activate Selected')
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->requiresConfirmation()
                ->action(function ($records) {
                    $records->each(function ($record) {
                        $record->profile()->update(['status' => UserProfileStatus::Active]);
                    });
                }),
            Tables\Actions\BulkAction::make('deactivateSelected')
                ->label('Put Selected On Hold')
                ->icon('heroicon-o-pause-circle')
                ->color('gray')
                ->requiresConfirmation()
                ->action(function ($records) {
                    $records->each(function ($record) {
                        $record->profile()->update(['status' => UserProfileStatus::OnHold]);
                    });
                }),
            Tables\Actions\BulkAction::make('cancelSelected')
                ->label('Cancel Selected')
                ->icon('heroicon-o-x-circle')
                ->color('danger')
                ->requiresConfirmation()
                ->action(function ($records) {
                    $records->each(function ($record) {
                        $record->profile()->update(['status' => UserProfileStatus::Cancelled]);
                    });
                }),
        ]);
    }
}
