<?php

namespace App\Filament\Resources\Member;

use App\Filament\Resources\Member\VideoResource\Pages;
use App\Filament\Resources\Member\VideoResource\RelationManagers;
use App\Models\User\UserVideo;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Illuminate\Support\Str;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
class VideoResource extends Resource
{
    protected static ?string $model = UserVideo::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 12,
                    ])
                    ->schema([
                        // Main Content Column
                        Group::make()
                            ->columnSpan(['lg' => 8])
                            ->schema([
                                Section::make('Video Information')
                                    ->schema([
                                        TextInput::make('video_title')
                                            ->required()
                                            ->maxLength(191),
                                        RichEditor::make('video_description')
                                            ->required()
                                            ->toolbarButtons([
                                                'bold',
                                                'italic',
                                                'underline',
                                                'strike',
                                                'link',
                                                'bulletList',
                                                'orderedList',
                                            ]),
                                        SpatieMediaLibraryFileUpload::make('video')
                                            ->collection('video')
                                            ->acceptedFileTypes(['video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov', 'video/wmv', 'video/flv'])
                                            ->required(),

                                    ]),
                            ]),
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                Section::make('Video Thumbnail')
                                    ->schema([
                                        TextInput::make('video_link')
                                            ->url()
                                            ->maxLength(255),
                                        Select::make('user_id')
                                            ->label('Belongs to')
                                            ->relationship('user', 'name')
                                            ->required()
                                            ->searchable()
                                            ->preload()
                                            ->helperText('Select the user who the video belongs to'),
                                        Toggle::make('is_active')
                                            ->required()
                                            ->default(true)
                                            ->helperText('Select if the video is active'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('video')
                    ->collection('video')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Video Thumbnail'),
                TextColumn::make('video_title')
                    ->searchable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 30))
                    ->tooltip(fn($state) => $state),
                TextColumn::make('video_link')
                    ->searchable()
                    ->formatStateUsing(fn ($state) => Str::limit($state, 30))
                    ->default('N/A')
                    ->tooltip(fn($state) => $state),
                IconColumn::make('is_active')
                    ->tooltip(fn($state) => $state),
                IconColumn::make('is_featured')
                    ->boolean()
                    ->trueIcon('heroicon-s-star')
                    ->falseIcon('heroicon-o-star')
                    ->trueColor('warning')
                    ->falseColor('gray')
                    ->label('Featured'),
                TextColumn::make('user.name')
                    ->searchable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 30))
                    ->tooltip(fn($state) => $state),
            ])
            ->filters([
                TrashedFilter::make(),
                SelectFilter::make('is_active')
                    ->options([
                        '1' => 'Active',
                        '0' => 'Inactive',
                    ]),
                SelectFilter::make('user_id')
                    ->label('Belongs to')
                    ->relationship('user', 'name'),
            ])
            ->actions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),
                    Tables\Actions\Action::make('feature')
                        ->icon('heroicon-o-star')
                        ->action(function (UserVideo $record) {
                            // First, unfeature any currently featured video
                            if (!$record->is_featured) {
                                UserVideo::where('is_featured', true)->update(['is_featured' => false]);
                                $record->update(['is_featured' => true]);
                            }
                        })
                        ->color(fn (UserVideo $record) => $record->is_featured ? 'warning' : 'gray')
                        ->icon(fn (UserVideo $record) => $record->is_featured ? 'heroicon-s-star' : 'heroicon-o-star')
                        ->tooltip(fn (UserVideo $record) => $record->is_featured ? 'This video is currently featured' : 'Mark as featured')
                        ->hidden(fn (UserVideo $record) => $record->is_featured),
                    DeleteAction::make(),
                ]),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListVideos::route('/'),
            'create' => Pages\CreateVideo::route('/create'),
            'edit' => Pages\EditVideo::route('/{record}/edit'),
        ];
    }
}
