<?php

namespace App\Filament\Resources\Membership;

use App\Filament\Resources\Membership\MembershipPlanResource\Pages;
use App\Filament\Resources\Membership\MembershipPlanResource\RelationManagers;
use App\Models\Membership\MembershipPlan;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Enums\MembershipType;
use App\Enums\MembershipPaymentTerm;
use App\Enums\AppModules;
use Filament\Forms\Get;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Tabs\Tab;
use Filament\Notifications\Notification;
use Illuminate\Support\Str;
use Filament\Forms\Set;

class MembershipPlanResource extends Resource
{
    protected static ?string $model = MembershipPlan::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // Adjusting to a 10-column grid for 70/30 split
                    ])
                    ->schema([
                        Group::make()
                            ->columnSpan(['lg' => 7])
                            ->schema([
                                Section::make('Basic Information')
                                    ->schema([
                                        Select::make('role_id')
                                            ->relationship('role', 'name')
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->live(),
                                        TextInput::make('package_title')
                                            ->label('Package Title')
                                            ->required()
                                            ->placeholder('Enter the title for this membership plan')
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                                if ($context === 'create' && empty($get('slug'))) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('slug')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (string $state, Set $set) {
                                                $set('slug', Str::slug($state));
                                            })
                                            ->dehydrated(),
                                        Select::make('membership_type')
                                            ->label('Membership Type')
                                            ->options(MembershipType::class)
                                            ->required()
                                            ->live(),
                                        Textarea::make('package_description')
                                            ->placeholder('Enter the description for this membership plan')
                                            ->columnSpanFull(),

                                    ])->columns(2),

                                Section::make('Pricing Details')
                                    ->collapsible()
                                    ->schema([
                                        TextInput::make('price_monthly')
                                            ->label('Price Monthly')
                                            ->placeholder('Enter the price for monthly membership')
                                            ->required()
                                            ->numeric()
                                            ->prefix('$')
                                            ->default(0.00)
                                            ->live(),
                                        TextInput::make('price_quarterly')
                                            ->label('Price Quarterly')
                                            ->placeholder('Enter the price for quarterly membership')
                                            ->required()
                                            ->numeric()
                                            ->prefix('$')
                                            ->default(0.00)
                                            ->live(),
                                        TextInput::make('price_half_yearly')
                                            ->label('Price Half Yearly')
                                            ->placeholder('Enter the price for half yearly membership')
                                            ->required()
                                            ->numeric()
                                            ->prefix('$')
                                            ->default(0.00)
                                            ->live(),
                                        TextInput::make('price_yearly')
                                            ->label('Price Yearly')
                                            ->placeholder('Enter the price for yearly membership')
                                            ->required()
                                            ->numeric()
                                            ->prefix('$')
                                            ->default(0.00)
                                            ->live(),
                                        Select::make('preferred_payment_term')
                                            ->label('Preferred Payment Term')
                                            ->options(MembershipPaymentTerm::class)
                                            ->required()
                                            ->default(MembershipPaymentTerm::Monthly)
                                            ->live(),
                                    ])->columns(2),

                                Section::make('Plan Access')
                                    ->collapsible()
                                    ->description('Configure the modules and features available to users with this plan')
                                    ->schema([
                                        Repeater::make('plan_access')
                                            ->relationship('planModules')
                                            ->label('Plan Modules')
                                            ->schema([
                                                Section::make()
                                                    ->compact()
                                                    ->schema([
                                                        Select::make('module_name')
                                                            ->options(AppModules::class)
                                                            ->required()
                                                            ->native(false)
                                                            ->distinct()
                                                            ->live(),
                                                        Toggle::make('module_allow_listing')
                                                            ->label('Can Post')
                                                            ->required()
                                                            ->live()
                                                            ->inline(false),

                                                        Grid::make(3)->schema([
                                                            TextInput::make('module_listing_limit')
                                                                ->label('Listing Limit')
                                                                ->numeric()
                                                                ->default(0)
                                                                ->visible(fn(Get $get) => $get('module_allow_listing')),
                                                            TextInput::make('module_allowed_images')
                                                                ->label('Images Per Listing')
                                                                ->numeric()
                                                                ->default(0)
                                                                ->visible(fn(Get $get) => $get('module_allow_listing')),
                                                            TextInput::make('module_extra_listing_price')
                                                                ->label('Extra Listing Price')
                                                                ->numeric()
                                                                ->default(0)
                                                                ->visible(fn(Get $get) => $get('module_allow_listing'))
                                                                ->prefix('$'),
                                                        ]),


                                                    ])->columns(2),
                                            ])
                                            ->itemLabel(
                                                fn(array $state): ?string =>
                                                $state['module_name'] ?? null
                                            )
                                            ->addActionLabel('Add Module')
                                            ->reorderableWithButtons()
                                            ->collapsible(),
                                    ]),




                                Section::make('Package Features')
                                    ->description('Configure what features will be available to users with this plan')
                                    ->collapsible()
                                    ->schema([
                                        Grid::make(1)->schema([
                                            Repeater::make('features')
                                                ->label('Features')
                                                ->schema([
                                                    TextInput::make('feature_name')
                                                        ->label('Feature Name')
                                                        ->helperText('Enter the name of the feature')
                                                        ->maxLength(100),
                                                ])
                                                ->itemLabel(
                                                    fn(array $state): ?string =>
                                                    $state['feature_name'] ?? null
                                                )
                                                ->addActionLabel('Add Feature')
                                                ->reorderableWithButtons()
                                                ->collapsible()
                                                ->collapsed()
                                                ->reorderable(),
                                        ]),
                                    ]),


                                Section::make('Package User Info')
                                    ->description('Configure what information and features will be visible on user profiles')
                                    ->relationship('packageUserInfo')
                                    ->collapsible()
                                    ->schema([
                                        Grid::make(2)->schema([
                                            Toggle::make('show_website_link')
                                                ->label('Website Link')
                                                ->helperText('Allow users to display their website')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('show_phone')
                                                ->label('Phone Number')
                                                ->helperText('Allow users to display their phone')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('show_address')
                                                ->label('Address')
                                                ->helperText('Allow users to display their address')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('show_social_media_links')
                                                ->label('Social Media')
                                                ->helperText('Allow users to display social media links')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('searchable')
                                                ->label('Profile Searchable')
                                                ->helperText('Make profile visible in search results')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('show_public_review')
                                                ->label('Public Reviews')
                                                ->helperText('Allow public reviews on profile')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Toggle::make('show_contact_form')
                                                ->label('Contact Form')
                                                ->helperText('Display contact form on profile')
                                                ->inline()
                                                ->required()
                                                ->live(),
                                            Grid::make(2)->schema([
                                                Group::make([
                                                    Toggle::make('show_image_gallery')
                                                        ->label('Enable Image Gallery')
                                                        ->helperText('Allow users to create image gallery')
                                                        ->inline()
                                                        ->required()
                                                        ->live(),
                                                    TextInput::make('image_gallery_limit')
                                                        ->label('Image Limit')
                                                        ->helperText('Maximum number of images allowed')
                                                        ->required()
                                                        ->numeric()
                                                        ->default(0)
                                                        ->visible(fn(Get $get) => $get('show_image_gallery')),
                                                ]),
                                                Group::make([
                                                    Toggle::make('show_video_gallery')
                                                        ->label('Enable Video Gallery')
                                                        ->helperText('Allow users to create video gallery')
                                                        ->inline()
                                                        ->required()
                                                        ->live(),
                                                    TextInput::make('video_gallery_limit')
                                                        ->label('Video Limit')
                                                        ->helperText('Maximum number of videos allowed')
                                                        ->required()
                                                        ->numeric()
                                                        ->default(0)
                                                        ->visible(fn(Get $get) => $get('show_video_gallery')),
                                                ]),
                                            ]),
                                        ]),
                                    ]),


                                // Section::make('Additional Costs')
                                //     ->schema([
                                //         Repeater::make('additional_costs')
                                //             ->relationship('additionalCosts')
                                //             ->schema([
                                //                 TextInput::make('title')
                                //                     ->required()
                                //                     ->maxLength(255),
                                //                 TextInput::make('price')
                                //                     ->required()
                                //                     ->numeric()
                                //                     ->prefix('$'),
                                //             ])->columns(2)
                                //             ->addActionLabel('Add Cost'),
                                //     ])
                            ]),

                        Group::make()
                            ->columnSpan(['lg' => 3])
                            ->schema([
                                Section::make('Plan Settings')
                                    ->collapsible()
                                    ->schema([
                                        Toggle::make('is_active')
                                            ->required()
                                            ->label('Active')
                                            ->helperText('Enable or disable this membership plan')
                                            ->live(),
                                        Toggle::make('is_visible')
                                            ->required()
                                            ->label('Visibility')
                                            ->helperText('Show or hide this plan on the frontend')
                                            ->live(),
                                        Toggle::make('is_featured')
                                            ->required()
                                            ->label('Set as Featured')
                                            ->helperText('Set this plan as featured')
                                            ->live(),
                                        Toggle::make('is_best_value')
                                            ->required()
                                            ->label('Set as Best Value')
                                            ->helperText('Set this plan as best value')
                                            ->live(),
                                        Toggle::make('is_most_popular')
                                            ->required()
                                            ->label('Set as Most Popular')
                                            ->helperText('Set this plan as most popular')
                                            ->live(),
                                        TextInput::make('display_order')
                                            ->required()
                                            ->numeric()
                                            ->default(0)
                                            ->helperText('Set the display order (lower numbers appear first)')
                                            ->live(),
                                    ]),

                                Section::make('Trial Settings')
                                    ->collapsible()
                                    ->schema([
                                        Toggle::make('has_trial')
                                            ->label('Enable Trial')
                                            ->required()
                                            ->helperText('Enable trial period for this plan')
                                            ->live(),
                                        TextInput::make('trial_days')
                                            ->required()
                                            ->label('Trial Days')
                                            ->numeric()
                                            ->helperText('Number of days for the trial period')
                                            ->default(0)
                                            ->visible(fn(Get $get) => $get('has_trial')),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('role.name')
                    ->label('Role')
                    ->sortable(),
                Tables\Columns\TextColumn::make('package_title')
                    ->label('Title')
                    ->searchable(),
                Tables\Columns\TextColumn::make('membership_type')
                    ->label('Type')
                    ->searchable(),
                Tables\Columns\TextColumn::make('prices')
                    ->label('Pricing')
                    ->html()
                    ->state(function (MembershipPlan $record): string {
                        // Show only monthly price by default
                        if ($record->price_monthly <= 0) {
                            return '<div class="text-sm text-gray-500">No price set</div>';
                        }

                        $allPrices = [];
                        if ($record->price_quarterly > 0) {
                            $allPrices[] = [
                                'period' => 'Quarterly',
                                'price' => $record->price_quarterly
                            ];
                        }
                        if ($record->price_half_yearly > 0) {
                            $allPrices[] = [
                                'period' => 'Biannually',
                                'price' => $record->price_half_yearly
                            ];
                        }
                        if ($record->price_yearly > 0) {
                            $allPrices[] = [
                                'period' => 'Annually',
                                'price' => $record->price_yearly
                            ];
                        }

                        $monthlyPrice = '<div class="text-sm text-gray-600 dark:text-gray-400">$' .
                            number_format($record->price_monthly, 2) . ' Monthly</div>';

                        if (!empty($allPrices)) {
                            $dropdownContent = '';
                            foreach ($allPrices as $price) {
                                $dropdownContent .= '<div class="px-3 py-1.5 text-sm text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700/50 rounded-md transition duration-150">$' .
                                    number_format($price['price'], 2) . ' ' . $price['period'] . '</div>';
                            }

                            return '
                                <div class="group relative inline-block text-start">
                                    ' . $monthlyPrice . '
                                    <div class="invisible absolute left-0 z-50 w-44 py-1.5 mt-1 bg-white dark:bg-gray-800 rounded-lg shadow-lg ring-1 ring-gray-200 dark:ring-gray-700 group-hover:visible transform transition duration-150 origin-top">
                                        ' . $dropdownContent . '
                                    </div>
                                </div>';
                        }

                        return $monthlyPrice;
                    }),
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success')
                    ->offColor('danger')
                    ->afterStateUpdated(function ($record, $state) {
                        Notification::make()
                            ->success()
                            ->title($state ? 'Plan Activated' : 'Plan Deactivated')
                            ->body($state ? 'The plan is now active.' : 'The plan has been deactivated.')
                            ->send();
                    }),
                Tables\Columns\ToggleColumn::make('is_visible')
                    ->label('Visible')
                    ->onColor('success')
                    ->offColor('warning')
                    ->afterStateUpdated(function ($record, $state) {
                        Notification::make()
                            ->success()
                            ->title($state ? 'Plan Visible' : 'Plan Hidden')
                            ->body($state ? 'The plan is now visible.' : 'The plan has been hidden.')
                            ->send();
                    }),
                Tables\Columns\TextColumn::make('subscriptions_count')
                    ->label('Members')
                    ->counts('subscriptions')
                    ->badge()
                    ->color(fn(int $state): string => match (true) {
                        $state === 0 => 'gray',
                        $state >= 10 => 'success',
                        default => 'info',
                    })
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('display_order')
                    ->label('Order')
                    ->numeric()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make()
                        ->hidden(fn(MembershipPlan $record): bool => $record->trashed()),
                    Tables\Actions\EditAction::make()
                        ->hidden(fn(MembershipPlan $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->hidden(fn(MembershipPlan $record): bool => $record->trashed()),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(MembershipPlan $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),

                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading();
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMembershipPlans::route('/'),
            'create' => Pages\CreateMembershipPlan::route('/create'),
            'view' => Pages\ViewMembershipPlan::route('/{record}'),
            'edit' => Pages\EditMembershipPlan::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
